<?php include 'includes/header.php'; ?>

<?php
require_once 'includes/mail.php';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = sanitize($_POST['email']);
    
    // Check if email exists in database
    $stmt = $conn->prepare("SELECT id, full_name, email FROM users WHERE email = ?");
    $stmt->execute([$email]);
    $user = $stmt->fetch();
    
    if ($user) {
        // Generate reset token
        $token = bin2hex(random_bytes(32));
        $expires = date('Y-m-d H:i:s', strtotime('+1 hour'));
        
        // Check if password_reset_tokens table exists, if not create it
        try {
            $conn->exec("CREATE TABLE IF NOT EXISTS password_reset_tokens (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL,
                token VARCHAR(64) NOT NULL,
                expires_at DATETIME NOT NULL,
                used BOOLEAN DEFAULT FALSE,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
                INDEX idx_token (token),
                INDEX idx_expires (expires_at)
            )");
        } catch (Exception $e) {
            // Table might already exist, continue
        }
        
        // Delete any existing tokens for this user
        $stmt = $conn->prepare("DELETE FROM password_reset_tokens WHERE user_id = ?");
        $stmt->execute([$user['id']]);
        
        // Insert new token
        $stmt = $conn->prepare("INSERT INTO password_reset_tokens (user_id, token, expires_at) VALUES (?, ?, ?)");
        $stmt->execute([$user['id'], $token, $expires]);
        
        // Send reset email
        $reset_link = SITE_URL . "reset-password.php?token=" . $token;
        
        $subject = "Password Reset Request - OACSOFT CBT System";
        $message = "
        <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #ddd; border-radius: 10px;'>
            <div style='text-align: center; margin-bottom: 30px;'>
                <h2 style='color: #2c3e50; margin-bottom: 10px;'>
                    <i class='fas fa-laptop-code' style='color: #3b82f6;'></i>
                    OACSOFT CBT System
                </h2>
                <p style='color: #666; margin: 0;'>Password Reset Request</p>
            </div>
            
            <div style='background: #f8f9fa; padding: 20px; border-radius: 8px; margin-bottom: 20px;'>
                <h3 style='color: #2c3e50; margin-top: 0;'>Hello " . htmlspecialchars($user['full_name']) . ",</h3>
                <p style='color: #555; line-height: 1.6;'>
                    We received a request to reset your password for your OACSOFT CBT System account. 
                    If you made this request, please click the button below to reset your password.
                </p>
            </div>
            
            <div style='text-align: center; margin: 30px 0;'>
                <a href='" . $reset_link . "' 
                   style='background: linear-gradient(135deg, #3b82f6, #1d4ed8); 
                          color: white; 
                          padding: 15px 30px; 
                          text-decoration: none; 
                          border-radius: 25px; 
                          font-weight: bold; 
                          display: inline-block;
                          box-shadow: 0 4px 15px rgba(59, 130, 246, 0.3);'>
                    Reset My Password
                </a>
            </div>
            
            <div style='background: #fff3cd; border: 1px solid #ffeaa7; padding: 15px; border-radius: 8px; margin: 20px 0;'>
                <p style='color: #856404; margin: 0; font-size: 14px;'>
                    <strong>Security Notice:</strong> This link will expire in 1 hour for your security. 
                    If you didn't request this password reset, please ignore this email.
                </p>
            </div>
            
            <div style='border-top: 1px solid #eee; padding-top: 20px; margin-top: 30px;'>
                <p style='color: #666; font-size: 12px; margin: 0;'>
                    If the button doesn't work, copy and paste this link into your browser:<br>
                    <a href='" . $reset_link . "' style='color: #3b82f6; word-break: break-all;'>" . $reset_link . "</a>
                </p>
            </div>
            
            <div style='text-align: center; margin-top: 30px; padding-top: 20px; border-top: 1px solid #eee;'>
                <p style='color: #999; font-size: 12px; margin: 0;'>
                    © " . date('Y') . " OACSOFT TECHNOLOGIES LTD. All rights reserved.
                </p>
            </div>
        </div>
        ";
        
        $mailHelper = new MailHelper();
        $emailSent = $mailHelper->sendEmail($user['email'], $user['full_name'], $subject, $message);
        
        if ($emailSent) {
            $success = "Password reset instructions have been sent to your email address.";
        } else {
            $error = "Failed to send reset email. Please try again later.";
        }
    } else {
        // Don't reveal if email exists or not for security
        $success = "If an account with that email exists, password reset instructions have been sent.";
    }
}
?>

<div class="login-container">
    <div class="card">
        <div class="card-header text-center">
            <h4 class="mb-0">
                <i class="fas fa-key me-2"></i>
                Forgot Password
            </h4>
        </div>
        <div class="card-body">
            <?php if (isset($success)): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle me-2"></i>
                    <?php echo $success; ?>
                </div>
                <div class="text-center mt-3">
                    <a href="login.php" class="btn btn-primary">
                        <i class="fas fa-arrow-left me-2"></i>
                        Back to Login
                    </a>
                </div>
            <?php else: ?>
                <?php if (isset($error)): ?>
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <?php echo $error; ?>
                    </div>
                <?php endif; ?>

                <div class="alert alert-info">
                    <i class="fas fa-info-circle me-2"></i>
                    Enter your email address and we'll send you instructions to reset your password.
                </div>

                <form method="POST">
                    <div class="mb-3">
                        <label for="email" class="form-label">
                            <i class="fas fa-envelope me-2"></i>
                            Email Address
                        </label>
                        <input type="email" class="form-control" id="email" name="email" required 
                               placeholder="Enter your registered email address">
                    </div>

                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-paper-plane me-2"></i>
                            Send Reset Instructions
                        </button>
                    </div>
                </form>

                <div class="text-center mt-3">
                    <a href="login.php" class="text-decoration-none">
                        <i class="fas fa-arrow-left me-2"></i>
                        Back to Login
                    </a>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
.login-container {
    min-height: 100vh;
    display: flex;
    align-items: center;
    justify-content: center;
    background: white;
    padding: 20px;
}

.card {
    width: 100%;
    max-width: 450px;
    border: none;
    border-radius: 15px;
    box-shadow: 0 15px 35px rgba(0, 0, 0, 0.1);
    overflow: hidden;
}

.card-header {
    background: linear-gradient(135deg, #3b82f6, #1d4ed8);
    color: white;
    padding: 2rem 1.5rem;
    border-bottom: none;
}

.card-body {
    padding: 2rem 1.5rem;
    background: white;
}

.form-control {
    border-radius: 10px;
    border: 2px solid #e2e8f0;
    padding: 12px 15px;
    transition: all 0.3s ease;
}

.form-control:focus {
    border-color: #3b82f6;
    box-shadow: 0 0 0 0.2rem rgba(59, 130, 246, 0.25);
}

.btn-primary {
    background: linear-gradient(135deg, #3b82f6, #1d4ed8);
    border: none;
    border-radius: 10px;
    padding: 12px 20px;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #1d4ed8, #1e40af);
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(59, 130, 246, 0.3);
}

.alert {
    border-radius: 10px;
    border: none;
    padding: 15px 20px;
}

.alert-success {
    background: linear-gradient(135deg, #10b981, #059669);
    color: white;
}

.alert-danger {
    background: linear-gradient(135deg, #ef4444, #dc2626);
    color: white;
}

.alert-info {
    background: linear-gradient(135deg, #3b82f6, #1d4ed8);
    color: white;
}

@media (max-width: 576px) {
    .login-container {
        padding: 10px;
    }
    
    .card-body {
        padding: 1.5rem 1rem;
    }
    
    .card-header {
        padding: 1.5rem 1rem;
    }
}
</style>

<?php include 'includes/footer.php'; ?>