-- OACSOFT TECHNOLOGIES LTD - Computer Based Test Database Schema

-- Create database
CREATE DATABASE IF NOT EXISTS oacsoft_cbt;
USE oacsoft_cbt;

-- Users table (for both candidates and admins)
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    full_name VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    phone VARCHAR(20),
    password VARCHAR(255) NOT NULL,
    role ENUM('candidate', 'admin') DEFAULT 'candidate',
    status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Job roles table
CREATE TABLE IF NOT EXISTS job_roles (
    id INT AUTO_INCREMENT PRIMARY KEY,
    role_name VARCHAR(100) UNIQUE NOT NULL,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Questions table
CREATE TABLE IF NOT EXISTS questions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    job_role_id INT NOT NULL,
    question TEXT NOT NULL,
    option_a VARCHAR(500) NOT NULL,
    option_b VARCHAR(500) NOT NULL,
    option_c VARCHAR(500) NOT NULL,
    option_d VARCHAR(500) NOT NULL,
    correct_answer ENUM('A', 'B', 'C', 'D') NOT NULL,
    marks INT DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (job_role_id) REFERENCES job_roles(id) ON DELETE CASCADE
);

-- Test sessions table
CREATE TABLE IF NOT EXISTS test_sessions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    job_role_id INT NOT NULL,
    access_code VARCHAR(10) UNIQUE NOT NULL,
    scheduled_time DATETIME,
    duration_minutes INT DEFAULT 30,
    status ENUM('scheduled', 'in_progress', 'completed', 'expired') DEFAULT 'scheduled',
    created_by INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (job_role_id) REFERENCES job_roles(id) ON DELETE CASCADE,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE CASCADE
);

-- Test results table
CREATE TABLE IF NOT EXISTS test_results (
    id INT AUTO_INCREMENT PRIMARY KEY,
    session_id INT NOT NULL,
    score INT NOT NULL,
    total_questions INT NOT NULL,
    correct_answers INT NOT NULL,
    percentage DECIMAL(5,2) NOT NULL,
    completed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    reviewed_by INT,
    reviewed_at TIMESTAMP NULL,
    email_sent BOOLEAN DEFAULT FALSE,
    FOREIGN KEY (session_id) REFERENCES test_sessions(id) ON DELETE CASCADE,
    FOREIGN KEY (reviewed_by) REFERENCES users(id) ON DELETE SET NULL
);

-- User answers table
CREATE TABLE IF NOT EXISTS user_answers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    session_id INT NOT NULL,
    question_id INT NOT NULL,
    selected_answer ENUM('A', 'B', 'C', 'D'),
    is_correct BOOLEAN,
    FOREIGN KEY (session_id) REFERENCES test_sessions(id) ON DELETE CASCADE,
    FOREIGN KEY (question_id) REFERENCES questions(id) ON DELETE CASCADE,
    UNIQUE KEY unique_session_question (session_id, question_id)
);

-- Insert default admin user (password: admin123)
INSERT INTO users (full_name, email, password, role, status) VALUES 
('System Administrator', 'admin@oacsoft.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin', 'approved');

-- Insert job roles
INSERT INTO job_roles (role_name, description) VALUES 
('Customer Service Representative', 'Responsible for handling customer inquiries, complaints, and providing excellent customer support'),
('Software Developer', 'Designs, develops, and maintains software applications'),
('Data Analyst', 'Analyzes data to provide insights and support decision-making'),
('Project Manager', 'Plans, executes, and manages projects to ensure successful delivery'),
('Quality Assurance Engineer', 'Ensures software quality through testing and validation');

-- Insert sample questions for Customer Service Representative (30 questions)
INSERT INTO questions (job_role_id, question, option_a, option_b, option_c, option_d, correct_answer, marks) VALUES
(1, 'What is the primary goal of customer service?', 'Increase sales', 'Resolve customer issues efficiently', 'Reduce costs', 'Promote products', 'B', 1),
(1, 'How should you handle an angry customer?', 'Argue with them', 'Listen actively and empathize', 'Transfer to supervisor immediately', 'End the call', 'B', 1),
(1, 'What does first-call resolution mean?', 'Resolving issue on first contact', 'Calling customer first', 'First attempt to help', 'Initial greeting', 'A', 1),
(1, 'Which is NOT a good customer service practice?', 'Active listening', 'Blaming other departments', 'Following up', 'Being patient', 'B', 1),
(1, 'What is the STAR method used for?', 'Sales techniques', 'Handling complaints', 'Structured response to behavioral questions', 'Time management', 'C', 1),
(1, 'How long should you typically put a customer on hold?', 'No more than 2 minutes', '5-10 minutes', 'As long as needed', '15 minutes', 'A', 1),
(1, 'What is upselling?', 'Selling expensive items', 'Selling additional products/services', 'Selling to new customers', 'Discount selling', 'B', 1),
(1, 'Which communication skill is most important?', 'Speaking clearly', 'Active listening', 'Using technical terms', 'Talking fast', 'B', 1),
(1, 'What should you do if you don\'t know the answer?', 'Guess the answer', 'Transfer to another agent', 'Admit you don\'t know and find out', 'Ignore the question', 'C', 1),
(1, 'What is CRM?', 'Customer Relationship Management', 'Call Resolution Method', 'Customer Response Management', 'Call Recording Machine', 'A', 1),
(1, 'How should you end a customer service call?', 'Quickly hang up', 'Ask if there\'s anything else you can help with', 'Say goodbye only', 'Transfer to survey', 'B', 1),
(1, 'What is empathy in customer service?', 'Feeling sorry for customers', 'Understanding and sharing customer feelings', 'Agreeing with complaints', 'Being emotional', 'B', 1),
(1, 'Which is a good way to build rapport?', 'Using customer\'s name', 'Speaking formally', 'Avoiding small talk', 'Keeping conversations short', 'A', 1),
(1, 'What should you document after a customer interaction?', 'Only complaints', 'Everything discussed', 'Only solutions', 'Customer personal opinions', 'B', 1),
(1, 'How do you handle unrealistic customer expectations?', 'Promise anything they want', 'Explain limitations clearly', 'Transfer to supervisor', 'End the interaction', 'B', 1),
(1, 'What is the purpose of a follow-up call?', 'Increase sales', 'Ensure customer satisfaction', 'Fill time', 'Meet quota', 'B', 1),
(1, 'Which tone is best for customer service?', 'Casual and friendly', 'Professional and helpful', 'Strict and formal', 'Enthusiastic and loud', 'B', 1),
(1, 'What is defusing a situation?', 'Making it worse', 'Calming an upset customer', 'Transferring the call', 'Taking a break', 'B', 1),
(1, 'How should you handle personal attacks from customers?', 'Take it personally', 'Stay professional and focus on issue', 'Hang up immediately', 'Argue back', 'B', 1),
(1, 'What is the benefit of using positive language?', 'Confuses customers', 'Creates better customer experience', 'Sounds fake', 'Takes longer', 'B', 1),
(1, 'When should you escalate an issue?', 'For every complaint', 'When you\'ve exhausted all options', 'Never', 'Only for VIP customers', 'B', 1),
(1, 'What is the purpose of a knowledge base?', 'Store customer data', 'Provide quick access to information', 'Track performance', 'Record calls', 'B', 1),
(1, 'How should you handle multiple customers at once?', 'Ignore some customers', 'Prioritize and communicate wait times', 'Rush through all', 'Ask them to call back', 'B', 1),
(1, 'What is the most important metric in customer service?', 'Call duration', 'Customer satisfaction', 'Number of calls', 'Sales made', 'B', 1),
(1, 'How should you respond to customer feedback?', 'Defend your actions', 'Thank them and use it for improvement', 'Ignore negative feedback', 'Blame the system', 'B', 1),
(1, 'What is the purpose of active listening?', 'To prepare your response', 'To truly understand customer needs', 'To fill silence', 'To show you\'re smart', 'B', 1),
(1, 'How should you handle confidential information?', 'Share with team members', 'Keep secure and private', 'Discuss openly', 'Use for personal advantage', 'B', 1),
(1, 'What is the benefit of using customer\'s name?', 'Shows you remember them', 'Personalizes the interaction', 'Required by policy', 'Fills time', 'B', 1),
(1, 'How should you handle technical issues during a call?', 'Blame the technology', 'Apologize and provide alternatives', 'End the call', 'Make customer wait', 'B', 1),
(1, 'What is the purpose of service level agreements (SLAs)?', 'Increase workload', 'Set clear expectations for service', 'Reduce quality', 'Complicate processes', 'B', 1),
(1, 'How should you handle a customer who refuses to provide information?', 'Insist strongly', 'Explain why it\'s needed and consequences', 'Give up', 'Transfer to supervisor', 'B', 1);