<?php 
include '../includes/header.php'; 

if (!isCandidate()) {
    redirect('../login.php');
}

// Check if test session is active
if (!isset($_SESSION['test_session'])) {
    $_SESSION['error_message'] = "No active test session found. Please start a test from your dashboard.";
    header('Location: dashboard.php');
    exit;
}

$session = $_SESSION['test_session'];
$session_id = $session['id'];
$job_role_id = $session['job_role_id'];

// Fetch questions for this job role
$stmt = $conn->prepare("SELECT * FROM questions WHERE job_role_id = ? ORDER BY id");
$stmt->execute([$job_role_id]);
$questions = $stmt->fetchAll();

if (count($questions) === 0) {
    $_SESSION['error_message'] = "No questions found for this job role. Please contact administrator.";
    header('Location: dashboard.php');
    exit;
}

// Handle answer submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_test'])) {
    $total_marks = 0;
    $score = 0;
    $answers = [];
    
    foreach ($questions as $question) {
        $question_id = $question['id'];
        $selected_answer = isset($_POST['question_' . $question_id]) ? $_POST['question_' . $question_id] : null;
        
        $is_correct = ($selected_answer === $question['correct_answer']);
        $question_score = $is_correct ? $question['marks'] : 0;
        
        $total_marks += $question['marks'];
        $score += $question_score;
        
        $answers[] = [
            'question_id' => $question_id,
            'selected_answer' => $selected_answer,
            'is_correct' => $is_correct,
            'score' => $question_score
        ];
    }
    
    // Calculate pass/fail (70% passing threshold)
    $percentage = ($score / $total_marks) * 100;
    $passed = $percentage >= 70;
    
    // Save results
    $conn->beginTransaction();
    try {
        // Insert result
        $correct_answers = count(array_filter($answers, function($a) { return $a['is_correct']; }));
        $stmt = $conn->prepare("INSERT INTO test_results (session_id, score, total_questions, correct_answers, percentage, completed_at) VALUES (?, ?, ?, ?, ?, NOW())");
        $stmt->execute([$session_id, $score, count($questions), $correct_answers, $percentage]);
        
        // Insert user answers
        $stmt = $conn->prepare("INSERT INTO user_answers (session_id, question_id, selected_answer, is_correct) VALUES (?, ?, ?, ?)");
        foreach ($answers as $answer) {
            $stmt->execute([$session_id, $answer['question_id'], $answer['selected_answer'], $answer['is_correct']]);
        }
        
        // Update session status
        $stmt = $conn->prepare("UPDATE test_sessions SET status = 'completed' WHERE id = ?");
        $stmt->execute([$session_id]);
        
        $conn->commit();
        
        // Clear test session
        unset($_SESSION['test_session']);
        unset($_SESSION['test_start_time']);
        unset($_SESSION['test_end_time']);
        
        $_SESSION['success_message'] = "Test completed successfully! Score: $score/$total_marks (" . round($percentage) . "%)";
        header('Location: result.php?id=' . $session_id);
        exit;
        
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error_message'] = "Error submitting test results. Please try again.";
        header('Location: dashboard.php');
        exit;
    }
}

// Auto-save answers (AJAX endpoint)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['auto_save'])) {
    $question_id = intval($_POST['question_id']);
    $answer = sanitizeInput($_POST['answer']);
    
    $_SESSION['auto_saved_answers'][$question_id] = $answer;
    
    echo json_encode(['success' => true]);
    exit;
}

$total_questions = count($questions);
$current_question = isset($_GET['q']) ? max(1, min($total_questions, intval($_GET['q']))) : 1;

// Get auto-saved answers
$auto_saved_answers = $_SESSION['auto_saved_answers'] ?? [];
?>

<div class="test-container">
    <!-- Test Header -->
    <div class="test-header">
        <div class="container-fluid">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <h4 class="mb-0"><?php echo htmlspecialchars($session['role_name']); ?> Test</h4>
                </div>
                <div class="col-md-6 text-end">
                    <div class="timer-container">
                        <i class="fas fa-clock"></i>
                        <span id="timer" class="timer-display">--:--</span>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Test Body -->
    <div class="test-body">
        <div class="container-fluid">
            <div class="row">
                <div class="col-lg-8">
                    <!-- Question Card -->
                    <div class="card question-card">
                        <div class="card-header">
                            <div class="d-flex justify-content-between align-items-center">
                                <span>Question <?php echo $current_question; ?> of <?php echo $total_questions; ?></span>
                                <span class="badge bg-primary"><?php echo $questions[$current_question-1]['marks']; ?> marks</span>
                            </div>
                        </div>
                        <div class="card-body">
                            <h5 class="question-text">
                                <?php echo htmlspecialchars($questions[$current_question-1]['question']); ?>
                            </h5>
                            
                            <div class="question-options mt-4">
                                <?php 
                                $question = $questions[$current_question-1];
                                $question_id = $question['id'];
                                $saved_answer = $auto_saved_answers[$question_id] ?? '';
                                
                                $options = [
                                    'A' => $question['option_a'],
                                    'B' => $question['option_b'],
                                    'C' => $question['option_c'],
                                    'D' => $question['option_d']
                                ];
                                
                                foreach ($options as $key => $option): 
                                    if (empty($option)) continue;
                                ?>
                                    <div class="form-check">
                                        <input class="form-check-input" 
                                               type="radio" 
                                               name="question_<?php echo $question_id; ?>" 
                                               id="option_<?php echo $key; ?>" 
                                               value="<?php echo $key; ?>"
                                               <?php echo $saved_answer === $key ? 'checked' : ''; ?>
                                               onchange="autoSaveAnswer(<?php echo $question_id; ?>, '<?php echo $key; ?>')">
                                        <label class="form-check-label" for="option_<?php echo $key; ?>">
                                            <strong><?php echo $key; ?>.</strong> 
                                            <?php echo htmlspecialchars($option); ?>
                                        </label>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Navigation Buttons -->
                    <div class="d-flex justify-content-between mt-3">
                        <button type="button" 
                                class="btn btn-secondary" 
                                onclick="changeQuestion(<?php echo $current_question-1; ?>)"
                                <?php echo $current_question <= 1 ? 'disabled' : ''; ?>>
                            <i class="fas fa-chevron-left me-1"></i>Previous
                        </button>
                        
                        <button type="button" 
                                class="btn btn-secondary" 
                                onclick="changeQuestion(<?php echo $current_question+1; ?>)"
                                <?php echo $current_question >= $total_questions ? 'disabled' : ''; ?>>
                            Next<i class="fas fa-chevron-right ms-1"></i>
                        </button>
                    </div>
                </div>
                
                <div class="col-lg-4">
                    <!-- Progress Card -->
                    <div class="card">
                        <div class="card-header">
                            <h6 class="mb-0">Progress</h6>
                        </div>
                        <div class="card-body">
                            <div class="progress mb-3">
                                <div class="progress-bar" 
                                     role="progressbar" 
                                     style="width: <?php echo ($current_question / $total_questions) * 100; ?>%">
                                    <?php echo round(($current_question / $total_questions) * 100); ?>%
                                </div>
                            </div>
                            
                            <!-- Question Navigation -->
                            <div class="question-nav">
                                <h6 class="mb-2">Questions</h6>
                                <div class="row g-2">
                                    <?php for ($i = 1; $i <= $total_questions; $i++): 
                                        $question_id = $questions[$i-1]['id'];
                                        $is_answered = isset($auto_saved_answers[$question_id]);
                                    ?>
                                        <div class="col-3">
                                            <a href="test.php?q=<?php echo $i; ?>" 
                                               class="btn btn-sm w-100 <?php echo $i === $current_question ? 'btn-primary' : ($is_answered ? 'btn-success' : 'btn-outline-secondary'); ?>">
                                                <?php echo $i; ?>
                                            </a>
                                        </div>
                                    <?php endfor; ?>
                                </div>
                            </div>
                            
                            <hr>
                            
                            <!-- Submit Test Button -->
                            <form method="POST" id="testForm" onsubmit="return confirmSubmit()">
                                <button type="submit" 
                                        name="submit_test" 
                                        class="btn btn-danger w-100">
                                    <i class="fas fa-check-circle me-2"></i>Submit Test
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.test-container {
    min-height: 100vh;
    display: flex;
    flex-direction: column;
}

.test-header {
    background: var(--primary-color);
    color: white;
    padding: 1rem 0;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.test-body {
    flex: 1;
    padding: 2rem 0;
}

.question-card {
    margin-bottom: 2rem;
}

.question-text {
    font-size: 1.2rem;
    line-height: 1.6;
}

.question-options .form-check {
    margin-bottom: 1rem;
    padding: 1rem;
    border: 1px solid #dee2e6;
    border-radius: 0.375rem;
    transition: all 0.3s ease;
}

.question-options .form-check:hover {
    background-color: #f8f9fa;
    border-color: var(--primary-color);
}

.question-options .form-check-input:checked + .form-check-label {
    color: var(--primary-color);
    font-weight: bold;
}

.timer-container {
    font-size: 1.5rem;
    font-weight: bold;
}

.timer-display {
    font-family: 'Courier New', monospace;
}

.question-nav .btn {
    font-size: 0.875rem;
}

@media (max-width: 768px) {
    .test-body {
        padding: 1rem 0;
    }
    
    .question-text {
        font-size: 1rem;
    }
}
</style>

<script>
let endTime = <?php echo $_SESSION['test_end_time'] * 1000; ?>;
let timerInterval;

function updateTimer() {
    const now = new Date().getTime();
    const timeLeft = endTime - now;
    
    if (timeLeft <= 0) {
        clearInterval(timerInterval);
        document.getElementById('timer').textContent = '00:00';
        Swal.fire({
                    icon: 'warning',
                    title: 'Time\'s Up!',
                    text: 'Your test will be submitted automatically.',
                    timer: 3000,
                    showConfirmButton: false
                });
        document.getElementById('testForm').submit();
        return;
    }
    
    const minutes = Math.floor(timeLeft / (1000 * 60));
    const seconds = Math.floor((timeLeft % (1000 * 60)) / 1000);
    
    document.getElementById('timer').textContent = 
        String(minutes).padStart(2, '0') + ':' + String(seconds).padStart(2, '0');
    
    // Change color when time is running low
    if (minutes < 5) {
        document.getElementById('timer').style.color = '#dc3545';
    }
}

function changeQuestion(questionNum) {
    if (questionNum >= 1 && questionNum <= <?php echo $total_questions; ?>) {
        window.location.href = 'test.php?q=' + questionNum;
    }
}

function autoSaveAnswer(questionId, answer) {
    fetch('test.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'auto_save=true&question_id=' + questionId + '&answer=' + answer
    });
}

function confirmSubmit() {
    const unanswered = <?php echo $total_questions; ?> - Object.keys(<?php echo json_encode($auto_saved_answers); ?>).length;
    
    if (unanswered > 0) {
        return confirm(`You have ${unanswered} unanswered question(s). Are you sure you want to submit?`);
    }
    
    return confirm('Are you sure you want to submit your test?');
}

// Initialize timer
updateTimer();
timerInterval = setInterval(updateTimer, 1000);

// Prevent accidental navigation
window.addEventListener('beforeunload', function(e) {
    e.preventDefault();
    e.returnValue = 'Are you sure you want to leave? Your test progress will be lost.';
});

// Disable right-click and F12
document.addEventListener('contextmenu', e => e.preventDefault());
document.addEventListener('keydown', function(e) {
    if (e.key === 'F12' || (e.ctrlKey && e.shiftKey && e.key === 'I')) {
        e.preventDefault();
    }
});
</script>

<?php include '../includes/footer.php'; ?>