<?php
session_start();
require_once '../config/database.php';

// Check if user is logged in and is a candidate
if (!isLoggedIn() || !isCandidate()) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

// Check if required data is provided
if (!isset($_POST['result_id']) || !isset($_POST['session_id'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Missing required data']);
    exit;
}

$result_id = (int)$_POST['result_id'];
$session_id = (int)$_POST['session_id'];
$user_id = $_SESSION['user_id'];

// Begin transaction
$conn->beginTransaction();

try {
    // Verify the result belongs to the session and is not completed
    $stmt = $conn->prepare("SELECT tr.*, ts.job_role_id FROM test_results tr JOIN test_sessions ts ON tr.session_id = ts.id WHERE tr.id = ? AND tr.session_id = ? AND tr.completed_at IS NULL");
    $stmt->execute([$result_id, $session_id]);
    $result = $stmt->fetch();

    if (!$result) {
        throw new Exception('Invalid test session');
    }

    // Get all questions for this job role
    $stmt = $conn->prepare("SELECT * FROM questions WHERE job_role_id = ?");
    $stmt->execute([$result['job_role_id']]);
    $questions = $stmt->fetchAll();

    $total_marks = 0;
    $score = 0;

    // Calculate score based on saved answers
    foreach ($questions as $question) {
        $stmt = $conn->prepare("SELECT * FROM user_answers WHERE result_id = ? AND question_id = ?");
        $stmt->execute([$result_id, $question['id']]);
        $user_answer = $stmt->fetch();

        $selected_answer = $user_answer ? $user_answer['selected_answer'] : null;
        $is_correct = ($selected_answer === $question['correct_answer']);
        $question_score = $is_correct ? $question['marks'] : 0;

        $total_marks += $question['marks'];
        $score += $question_score;

        // Update the answer with correctness and score
        $stmt = $conn->prepare("UPDATE user_answers SET is_correct = ?, score = ? WHERE result_id = ? AND question_id = ?");
        $stmt->execute([$is_correct, $question_score, $result_id, $question['id']]);
    }

    // Calculate percentage and determine pass/fail
    $percentage = $total_marks > 0 ? ($score / $total_marks) * 100 : 0;
    $passed = $percentage >= 70; // 70% passing threshold

    // Update result with final scores
    $stmt = $conn->prepare("UPDATE test_results SET score = ?, total_marks = ?, percentage = ?, passed = ?, completed_at = NOW() WHERE id = ?");
    $stmt->execute([$score, $total_marks, $percentage, $passed, $result_id]);

    // Update test session status
    $stmt = $conn->prepare("UPDATE test_sessions SET status = 'completed' WHERE id = ?");
    $stmt->execute([$session_id]);

    // Get user details for email
    $stmt = $conn->prepare("SELECT u.email, u.full_name as name, jr.role_name FROM users u JOIN test_sessions ts ON u.id = ts.user_id JOIN job_roles jr ON jr.id = ts.job_role_id WHERE ts.id = ?");
    $stmt->execute([$session_id]);
    $user = $stmt->fetch();

    // Commit transaction
    $conn->commit();

    // Clean up session variables
    unset($_SESSION['test_started']);
    unset($_SESSION['result_id']);
    unset($_SESSION['test_start_time']);

    // Send email notification using SMTP
    if ($user) {
        require_once '../includes/mail.php';
        
        $subject = "CBT Test Results - " . $user['role_name'];
        $status = $passed ? 'PASSED' : 'FAILED';
        $status_color = $passed ? '#28a745' : '#dc3545';
        
        $message = "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <title>Test Results</title>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #007bff; color: white; padding: 20px; text-align: center; }
                .content { padding: 20px; background: #f8f9fa; }
                .result-box { background: white; padding: 15px; margin: 10px 0; border-radius: 5px; border-left: 4px solid #007bff; }
                .status { font-weight: bold; color: $status_color; }
                .footer { text-align: center; padding: 20px; color: #666; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>OACSOFT CBT System</h1>
                    <p>Test Completion Notification</p>
                </div>
                <div class='content'>
                    <h2>Dear {$user['name']},</h2>
                    <p>Your test for <strong>{$user['role_name']}</strong> has been completed successfully.</p>
                    
                    <div class='result-box'>
                        <h3>Test Results Summary</h3>
                        <p><strong>Score:</strong> $score out of $total_marks</p>
                        <p><strong>Percentage:</strong> " . round($percentage) . "%</p>
                        <p><strong>Status:</strong> <span class='status'>$status</span></p>
                        <p><strong>Completed At:</strong> " . date('M d, Y H:i:s') . "</p>
                    </div>
                    
                    <p>You can log in to your dashboard to view detailed results.</p>
                    <p>Your results are being reviewed and you will receive a detailed report soon.</p>
                </div>
                <div class='footer'>
                    <p>Best regards,<br>OACSOFT CBT Team</p>
                    <p><small>This is an automated message. Please do not reply to this email.</small></p>
                </div>
            </div>
        </body>
        </html>
        ";
        
        $mailHelper = new MailHelper();
        $email_result = $mailHelper->sendEmail($user['email'], $user['name'], $subject, $message, true);
        
        // Log email result (optional)
        if (!$email_result['success']) {
            error_log("Failed to send test completion email to {$user['email']}: " . $email_result['message']);
        }
    }

    echo json_encode([
        'success' => true,
        'result_id' => $result_id,
        'score' => $score,
        'total_marks' => $total_marks,
        'percentage' => round($percentage, 2),
        'passed' => $passed
    ]);

} catch (Exception $e) {
    $conn->rollBack();
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
?>