<?php 
include '../includes/header.php'; 

if (!isCandidate()) {
    redirect('../login.php');
}

$user_id = $_SESSION['user_id'];

// Handle test start request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['access_code'])) {
    $access_code = sanitizeInput($_POST['access_code']);
    
    // Verify access code and get session details
    $stmt = $conn->prepare("SELECT ts.*, jr.role_name 
                           FROM test_sessions ts 
                           JOIN job_roles jr ON ts.job_role_id = jr.id 
                           WHERE ts.access_code = ? AND ts.user_id = ? AND ts.status = 'scheduled'");
    $stmt->execute([$access_code, $user_id]);
    $session = $stmt->fetch();
    
    if ($session) {
        // Check if test has already been started
        $stmt = $conn->prepare("SELECT r.* FROM test_results r WHERE r.session_id = ?");
        $stmt->execute([$session['id']]);
        $existing_result = $stmt->fetch();
        
        if ($existing_result) {
            $_SESSION['error_message'] = "You have already completed this test.";
        } else {
            // Check scheduled time
            $scheduled_time = strtotime($session['scheduled_time']);
            $current_time = time();
            
            if ($current_time < $scheduled_time) {
                $_SESSION['error_message'] = "Test has not started yet. Scheduled time: " . date('Y-m-d H:i', $scheduled_time);
            } else {
                // Start the test
                $_SESSION['test_session'] = $session;
                $_SESSION['test_start_time'] = time();
                $_SESSION['test_end_time'] = time() + ($session['duration_minutes'] * 60);
                
                // Update session status
                $stmt = $conn->prepare("UPDATE test_sessions SET status = 'in_progress' WHERE id = ?");
                $stmt->execute([$session['id']]);
                
                header('Location: test.php');
                exit;
            }
        }
    } else {
        $_SESSION['error_message'] = "Invalid access code or test session not found.";
    }
}

// Fetch user's test sessions
$stmt = $conn->prepare("SELECT ts.*, jr.role_name, 
                       (SELECT COUNT(*) FROM questions WHERE job_role_id = ts.job_role_id) as total_questions
                       FROM test_sessions ts 
                       JOIN job_roles jr ON ts.job_role_id = jr.id 
                       WHERE ts.user_id = ? 
                       ORDER BY ts.created_at DESC");
$stmt->execute([$user_id]);
$sessions = $stmt->fetchAll();

// Fetch test results
$stmt = $conn->prepare("SELECT r.*, jr.role_name, ts.scheduled_time, ts.duration_minutes
                       FROM test_results r 
                       JOIN test_sessions ts ON r.session_id = ts.id 
                       JOIN job_roles jr ON ts.job_role_id = jr.id 
                       WHERE ts.user_id = ? 
                       ORDER BY r.completed_at DESC");
$stmt->execute([$user_id]);
$results = $stmt->fetchAll();

// Get user details
$stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$user_id]);
$user = $stmt->fetch();

if (!$user) {
    $_SESSION['error_message'] = "User not found. Please contact administrator.";
    redirect('../login.php');
}
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h2>
            <i class="fas fa-user-circle me-2"></i>
            Candidate Dashboard
        </h2>
        <p class="text-muted mb-0">Welcome back, <?php echo htmlspecialchars($user['full_name'] ?? 'User'); ?>!</p>
    </div>
    <div>
        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#startTestModal">
            <i class="fas fa-play me-2"></i>Start Test
        </button>
    </div>
</div>

<?php if (isset($_SESSION['success_message'])): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?php echo $_SESSION['success_message']; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php unset($_SESSION['success_message']); ?>
<?php endif; ?>

<?php if (isset($_SESSION['error_message'])): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?php echo $_SESSION['error_message']; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php unset($_SESSION['error_message']); ?>
<?php endif; ?>

<!-- Quick Actions -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="fas fa-bolt me-2"></i>
                    Quick Actions
                </h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6 mb-2">
                        <a href="test-history.php" class="btn btn-outline-info w-100">
                            <i class="fas fa-history me-2"></i>
                            View Test History
                        </a>
                    </div>
                    <div class="col-md-6 mb-2">
                        <button class="btn btn-outline-primary w-100" data-bs-toggle="modal" data-bs-target="#startTestModal">
                            <i class="fas fa-play me-2"></i>
                            Start New Test
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- Test Sessions Card -->
    <div class="col-lg-6 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="fas fa-clock me-2"></i>
                    Your Test Sessions
                </h5>
            </div>
            <div class="card-body">
                <?php if (count($sessions) > 0): ?>
                    <div class="list-group">
                        <?php foreach ($sessions as $session): 
                            $scheduled_time = strtotime($session['scheduled_time']);
                            $is_ready = $scheduled_time <= time() && $session['status'] === 'scheduled';
                            $is_past = $scheduled_time < time();
                        ?>
                            <div class="list-group-item">
                                <div class="d-flex justify-content-between align-items-start">
                                    <div class="flex-grow-1">
                                        <h6 class="mb-1"><?php echo htmlspecialchars($session['role_name']); ?></h6>
                                        <p class="mb-1">
                                            <small class="text-muted">
                                                <?php echo date('M j, Y H:i', $scheduled_time); ?> | 
                                                <?php echo $session['duration_minutes'] ?? '30'; ?> min | 
                                                <?php echo $session['total_questions'] ?? '0'; ?> questions
                                            </small>
                                        </p>
                                        <span class="badge status-<?php echo $session['status']; ?>">
                                            <?php echo ucfirst($session['status']); ?>
                                        </span>
                                    </div>
                                    <div class="ms-2">
                                        <?php if ($is_ready): ?>
                                            <button class="btn btn-sm btn-success" 
                                                    onclick="startTestWithCode('<?php echo $session['access_code']; ?>')">
                                                <i class="fas fa-play"></i>
                                            </button>
                                        <?php elseif ($session['status'] === 'active'): ?>
                                            <a href="test.php" class="btn btn-sm btn-warning">
                                                <i class="fas fa-arrow-right"></i>
                                            </a>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="text-center py-4">
                        <i class="fas fa-clock fa-3x text-muted mb-3"></i>
                        <h6 class="text-muted">No test sessions scheduled</h6>
                        <p class="text-muted">Wait for an admin to schedule a test for you.</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Test Results Card -->
    <div class="col-lg-6 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="fas fa-chart-line me-2"></i>
                    Your Test Results
                </h5>
            </div>
            <div class="card-body">
                <?php if (count($results) > 0): ?>
                    <div class="list-group">
                        <?php foreach ($results as $result): 
                            $score = $result['score'] ?? 0;
                            $total_marks = $result['total_questions'] ?? 1;
                            $percentage = round(($score / $total_marks) * 100);
                        ?>
                            <div class="list-group-item">
                                <div class="d-flex justify-content-between align-items-start">
                                    <div class="flex-grow-1">
                                        <h6 class="mb-1"><?php echo htmlspecialchars($result['role_name']); ?></h6>
                                        <p class="mb-1">
                                            <small class="text-muted">
                                                <?php echo date('M j, Y H:i', strtotime($result['completed_at'] ?? 'now')); ?> | 
                                                <?php echo $result['duration_minutes'] ?? '30'; ?> min
                                            </small>
                                        </p>
                                        <div class="d-flex align-items-center">
                                            <span class="badge bg-<?php echo $percentage >= 70 ? 'success' : ($percentage >= 50 ? 'warning' : 'danger'); ?>">
                                                <?php echo $result['score'] ?? 0; ?>/<?php echo $result['total_questions'] ?? 0; ?> (<?php echo $percentage; ?>%)
                                            </span>
                                            <?php if (($result['percentage'] ?? 0) >= 70): ?>
                                                <span class="badge bg-success ms-2">
                                                    <i class="fas fa-check"></i> Passed
                                                </span>
                                            <?php else: ?>
                                                <span class="badge bg-danger ms-2">
                                                    <i class="fas fa-times"></i> Failed
                                                </span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div class="ms-2">
                                        <a href="result.php?id=<?php echo $result['session_id']; ?>" 
                                           class="btn btn-sm btn-primary">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="text-center py-4">
                        <i class="fas fa-chart-line fa-3x text-muted mb-3"></i>
                        <h6 class="text-muted">No test results yet</h6>
                        <p class="text-muted">Complete a test to see your results here.</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Start Test Modal -->
<div class="modal fade" id="startTestModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Start Test</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="access_code" class="form-label">Access Code *</label>
                        <input type="text" 
                               class="form-control" 
                               id="access_code" 
                               name="access_code" 
                               required
                               placeholder="Enter the access code provided by admin">
                    </div>
                    <div class="alert alert-info">
                        <small>
                            <i class="fas fa-info-circle me-1"></i>
                            Contact your administrator if you don't have an access code.
                        </small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Start Test</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function startTestWithCode(accessCode) {
    document.getElementById('access_code').value = accessCode;
    document.querySelector('form').submit();
}

// Auto-focus on access code input
const startTestModal = document.getElementById('startTestModal');
startTestModal.addEventListener('shown.bs.modal', function () {
    document.getElementById('access_code').focus();
});
</script>

<?php include '../includes/footer.php'; ?>