// OACSOFT CBT System JavaScript

// Global variables
let examTimer;
let timeLeft;
let currentQuestion = 1;
let totalQuestions;

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    initializeApp();
});

function initializeApp() {
    // Initialize tooltips
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });

    // Initialize form validation
    initializeFormValidation();
    
    // Initialize test timer if on test page
    if (document.getElementById('testTimer')) {
        initializeTestTimer();
    }
    
    // Initialize auto-save for test answers
    if (document.querySelector('.test-form')) {
        initializeAutoSave();
    }
}

// Form validation
function initializeFormValidation() {
    const forms = document.querySelectorAll('.needs-validation');
    Array.prototype.slice.call(forms).forEach(function (form) {
        form.addEventListener('submit', function (event) {
            if (!form.checkValidity()) {
                event.preventDefault();
                event.stopPropagation();
            }
            form.classList.add('was-validated');
        }, false);
    });
}

// Test timer functionality
function initializeTestTimer() {
    const timerElement = document.getElementById('testTimer');
    const duration = parseInt(timerElement.dataset.duration) * 60; // Convert minutes to seconds
    
    timeLeft = duration;
    totalQuestions = parseInt(document.querySelector('.test-form').dataset.totalQuestions);
    
    updateTimerDisplay();
    examTimer = setInterval(updateTimer, 1000);
}

function updateTimer() {
    timeLeft--;
    updateTimerDisplay();
    
    if (timeLeft <= 0) {
        clearInterval(examTimer);
        autoSubmitTest();
    }
    
    // Warning at 5 minutes remaining
    if (timeLeft === 300) {
        showWarning('Warning: 5 minutes remaining!');
    }
}

function updateTimerDisplay() {
    const timerElement = document.getElementById('testTimer');
    if (!timerElement) return;
    
    const hours = Math.floor(timeLeft / 3600);
    const minutes = Math.floor((timeLeft % 3600) / 60);
    const seconds = timeLeft % 60;
    
    timerElement.innerHTML = `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
    
    // Change color when time is running low
    if (timeLeft <= 300) {
        timerElement.style.color = '#dc3545';
    }
}

// Auto-save test answers
function initializeAutoSave() {
    const answerInputs = document.querySelectorAll('input[type="radio"]');
    answerInputs.forEach(input => {
        input.addEventListener('change', function() {
            saveAnswer(this.name, this.value);
        });
    });
}

function saveAnswer(questionId, answer) {
    const sessionId = document.querySelector('.test-form').dataset.sessionId;
    
    fetch('ajax/save-answer.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            session_id: sessionId,
            question_id: questionId,
            answer: answer
        })
    })
    .then(response => response.json())
    .then(data => {
        if (!data.success) {
            console.error('Failed to save answer');
        }
    })
    .catch(error => console.error('Error:', error));
}

// Auto-submit test when time expires
function autoSubmitTest() {
    const form = document.querySelector('.test-form');
    if (form) {
        showWarning('Time\'s up! Submitting your test...');
        setTimeout(() => form.submit(), 2000);
    }
}

// Navigation through questions
function nextQuestion() {
    if (currentQuestion < totalQuestions) {
        document.getElementById(`question-${currentQuestion}`).style.display = 'none';
        currentQuestion++;
        document.getElementById(`question-${currentQuestion}`).style.display = 'block';
        updateProgressBar();
    }
}

function prevQuestion() {
    if (currentQuestion > 1) {
        document.getElementById(`question-${currentQuestion}`).style.display = 'none';
        currentQuestion--;
        document.getElementById(`question-${currentQuestion}`).style.display = 'block';
        updateProgressBar();
    }
}

function jumpToQuestion(questionNumber) {
    if (questionNumber >= 1 && questionNumber <= totalQuestions) {
        document.getElementById(`question-${currentQuestion}`).style.display = 'none';
        currentQuestion = questionNumber;
        document.getElementById(`question-${currentQuestion}`).style.display = 'block';
        updateProgressBar();
    }
}

function updateProgressBar() {
    const progressBar = document.querySelector('.progress-bar');
    if (progressBar) {
        const percentage = (currentQuestion / totalQuestions) * 100;
        progressBar.style.width = `${percentage}%`;
        progressBar.setAttribute('aria-valuenow', currentQuestion);
        progressBar.textContent = `${currentQuestion} of ${totalQuestions}`;
    }
}

// AJAX functions for admin panel
function approveUser(userId) {
    if (confirm('Are you sure you want to approve this user?')) {
        fetch('ajax/approve-user.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ user_id: userId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                Swal.fire({
                        icon: 'error',
                        title: 'Error',
                        text: data.message
                    });
            }
        })
        .catch(error => console.error('Error:', error));
    }
}

function rejectUser(userId) {
    if (confirm('Are you sure you want to reject this user?')) {
        fetch('ajax/reject-user.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ user_id: userId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: data.message
                });
            }
        })
        .catch(error => console.error('Error:', error));
    }
}

// Delete functionality with confirmation
function deleteItem(type, id) {
    const message = `Are you sure you want to delete this ${type}? This action cannot be undone.`;
    if (confirm(message)) {
        fetch(`ajax/delete-${type}.php`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ id: id })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: data.message
                });
            }
        })
        .catch(error => console.error('Error:', error));
    }
}

// Search functionality
function searchTable(inputId, tableId) {
    const input = document.getElementById(inputId);
    const table = document.getElementById(tableId);
    
    if (!input || !table) return;
    
    input.addEventListener('keyup', function() {
        const filter = this.value.toLowerCase();
        const rows = table.getElementsByTagName('tr');
        
        for (let i = 1; i < rows.length; i++) {
            const row = rows[i];
            const text = row.textContent.toLowerCase();
            row.style.display = text.includes(filter) ? '' : 'none';
        }
    });
}

// Email sending functionality
function sendResultEmail(resultId) {
    if (confirm('Send test results to candidate\'s email?')) {
        const button = event.target;
        button.innerHTML = '<span class="loading-spinner"></span> Sending...';
        button.disabled = true;
        
        fetch('ajax/send-result-email.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ result_id: resultId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showSuccess('Email sent successfully!');
                setTimeout(() => location.reload(), 2000);
            } else {
                showError('Failed to send email: ' + data.message);
                button.innerHTML = '<i class="fas fa-envelope"></i> Send Email';
                button.disabled = false;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showError('An error occurred while sending the email.');
            button.innerHTML = '<i class="fas fa-envelope"></i> Send Email';
            button.disabled = false;
        });
    }
}

// Utility functions
function showSuccess(message) {
    showToast(message, 'success');
}

function showError(message) {
    showToast(message, 'danger');
}

function showWarning(message) {
    showToast(message, 'warning');
}

function showToast(message, type = 'info') {
    const toastContainer = document.createElement('div');
    toastContainer.className = 'toast-container position-fixed bottom-0 end-0 p-3';
    toastContainer.innerHTML = `
        <div class="toast show" role="alert">
            <div class="toast-header bg-${type} text-white">
                <strong class="me-auto">${type.charAt(0).toUpperCase() + type.slice(1)}</strong>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="toast"></button>
            </div>
            <div class="toast-body">
                ${message}
            </div>
        </div>
    `;
    
    document.body.appendChild(toastContainer);
    
    const toast = new bootstrap.Toast(toastContainer.querySelector('.toast'));
    toast.show();
    
    setTimeout(() => toastContainer.remove(), 5000);
}

// Password strength indicator
function checkPasswordStrength(password) {
    const strengthBar = document.getElementById('passwordStrength');
    const strengthText = document.getElementById('passwordStrengthText');
    
    if (!strengthBar || !strengthText) return;
    
    let strength = 0;
    
    if (password.length >= 8) strength++;
    if (/[a-z]/.test(password)) strength++;
    if (/[A-Z]/.test(password)) strength++;
    if (/[0-9]/.test(password)) strength++;
    if (/[^A-Za-z0-9]/.test(password)) strength++;
    
    const colors = ['danger', 'warning', 'info', 'primary', 'success'];
    const texts = ['Very Weak', 'Weak', 'Fair', 'Good', 'Strong'];
    
    strengthBar.className = `progress-bar bg-${colors[strength]}`;
    strengthBar.style.width = `${(strength / 5) * 100}%`;
    strengthText.textContent = texts[strength];
}

// Initialize search functionality on relevant pages
document.addEventListener('DOMContentLoaded', function() {
    if (document.getElementById('searchUsers')) {
        searchTable('searchUsers', 'usersTable');
    }
    if (document.getElementById('searchQuestions')) {
        searchTable('searchQuestions', 'questionsTable');
    }
    if (document.getElementById('searchResults')) {
        searchTable('searchResults', 'resultsTable');
    }
});