<?php 
include '../includes/header.php'; 

if (!isAdmin()) {
    redirect('../login.php');
}

// Get result ID from URL
$result_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

if (!$result_id) {
    $_SESSION['error_message'] = 'Invalid result ID';
    header('Location: results.php');
    exit;
}

// Fetch test result details with related information
$stmt = $conn->prepare("
    SELECT 
        tr.*,
        ts.access_code,
        ts.scheduled_time,
        ts.duration_minutes,
        ts.status as session_status,
        u.full_name,
        u.email,
        u.phone,
        jr.role_name,
        jr.description as job_description,
        reviewer.full_name as reviewer_name
    FROM test_results tr
    JOIN test_sessions ts ON tr.session_id = ts.id
    JOIN users u ON ts.user_id = u.id
    JOIN job_roles jr ON ts.job_role_id = jr.id
    LEFT JOIN users reviewer ON tr.reviewed_by = reviewer.id
    WHERE tr.id = ?
");
$stmt->execute([$result_id]);
$result = $stmt->fetch();

if (!$result) {
    $_SESSION['error_message'] = 'Test result not found';
    header('Location: results.php');
    exit;
}

// Pagination for questions
$questions_per_page = 5;
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$offset = ($page - 1) * $questions_per_page;

// Get total number of questions
$count_stmt = $conn->prepare("
    SELECT COUNT(*) as total
    FROM questions q
    WHERE q.job_role_id = (
        SELECT job_role_id FROM test_sessions WHERE id = ?
    )
");
$count_stmt->execute([$result['session_id']]);
$total_questions = $count_stmt->fetch()['total'];
$total_pages = ceil($total_questions / $questions_per_page);

// Fetch questions and answers for this test with pagination
$stmt = $conn->prepare("
    SELECT 
        q.id,
        q.question,
        q.option_a,
        q.option_b,
        q.option_c,
        q.option_d,
        q.correct_answer,
        q.marks,
        ua.selected_answer,
        ua.is_correct
    FROM questions q
    LEFT JOIN user_answers ua ON q.id = ua.question_id AND ua.session_id = ?
    WHERE q.job_role_id = (
        SELECT job_role_id FROM test_sessions WHERE id = ?
    )
    ORDER BY q.id
    LIMIT ? OFFSET ?
");
$stmt->execute([$result['session_id'], $result['session_id'], $questions_per_page, $offset]);
$questions = $stmt->fetchAll();

// Calculate time taken (approximate based on scheduled time and completion)
$time_taken = '';
if ($result['scheduled_time'] && $result['completed_at']) {
    $start = new DateTime($result['scheduled_time']);
    $end = new DateTime($result['completed_at']);
    $diff = $start->diff($end);
    $time_taken = $diff->format('%H:%I:%S');
}

// Handle actions
if ($_POST) {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'mark_reviewed':
                $stmt = $conn->prepare("UPDATE test_results SET reviewed_by = ?, reviewed_at = NOW() WHERE id = ?");
                $stmt->execute([$_SESSION['user_id'], $result_id]);
                $_SESSION['success_message'] = 'Result marked as reviewed successfully!';
                header('Location: view-result.php?id=' . $result_id);
                exit;
                break;
                
            case 'send_email':
                $stmt = $conn->prepare("UPDATE test_results SET email_sent = TRUE WHERE id = ?");
                $stmt->execute([$result_id]);
                $_SESSION['success_message'] = 'Email status updated successfully!';
                header('Location: view-result.php?id=' . $result_id);
                exit;
                break;
        }
    }
}
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h2><i class="fas fa-chart-line me-2"></i>Test Result Details</h2>
    <a href="results.php" class="btn btn-secondary">
        <i class="fas fa-arrow-left me-1"></i>Back to Results
    </a>
</div>

<?php if (isset($_SESSION['success_message'])): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?php echo $_SESSION['success_message']; unset($_SESSION['success_message']); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<!-- Candidate Information -->
<div class="row mb-4">
    <div class="col-md-6">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="fas fa-user me-2"></i>Candidate Information</h5>
            </div>
            <div class="card-body">
                <table class="table table-borderless">
                    <tr>
                        <td><strong>Name:</strong></td>
                        <td><?php echo htmlspecialchars($result['full_name']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Email:</strong></td>
                        <td><?php echo htmlspecialchars($result['email']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Phone:</strong></td>
                        <td><?php echo htmlspecialchars($result['phone'] ?? 'N/A'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Job Role:</strong></td>
                        <td><?php echo htmlspecialchars($result['role_name']); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Access Code:</strong></td>
                        <td><span class="badge bg-info"><?php echo htmlspecialchars($result['access_code']); ?></span></td>
                    </tr>
                </table>
            </div>
        </div>
    </div>
    
    <div class="col-md-6">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="fas fa-clock me-2"></i>Test Information</h5>
            </div>
            <div class="card-body">
                <table class="table table-borderless">
                    <tr>
                        <td><strong>Scheduled Time:</strong></td>
                        <td><?php echo date('M d, Y H:i:s', strtotime($result['scheduled_time'])); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Completed At:</strong></td>
                        <td><?php echo date('M d, Y H:i:s', strtotime($result['completed_at'])); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Time Taken:</strong></td>
                        <td><?php echo $time_taken ?: 'N/A'; ?></td>
                    </tr>
                    <tr>
                        <td><strong>Duration:</strong></td>
                        <td><?php echo $result['duration_minutes']; ?> minutes</td>
                    </tr>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Performance Summary -->
<div class="row mb-4">
    <div class="col-md-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="fas fa-chart-bar me-2"></i>Performance Summary</h5>
            </div>
            <div class="card-body">
                <div class="row text-center">
                    <div class="col-md-2">
                        <div class="border rounded p-3">
                            <h4 class="text-primary"><?php echo $result['score']; ?></h4>
                            <small class="text-muted">Total Score</small>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="border rounded p-3">
                            <h4 class="text-success"><?php echo $result['correct_answers']; ?></h4>
                            <small class="text-muted">Correct</small>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="border rounded p-3">
                            <h4 class="text-danger"><?php echo $result['total_questions'] - $result['correct_answers']; ?></h4>
                            <small class="text-muted">Incorrect</small>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="border rounded p-3">
                            <h4 class="text-info"><?php echo $result['total_questions']; ?></h4>
                            <small class="text-muted">Total Questions</small>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="border rounded p-3">
                            <h4 class="<?php echo $result['percentage'] >= 70 ? 'text-success' : 'text-warning'; ?>">
                                <?php echo number_format($result['percentage'], 1); ?>%
                            </h4>
                            <small class="text-muted">Percentage</small>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="border rounded p-3">
                            <h4 class="<?php echo $result['percentage'] >= 70 ? 'text-success' : 'text-danger'; ?>">
                                <?php echo $result['percentage'] >= 70 ? 'PASS' : 'FAIL'; ?>
                            </h4>
                            <small class="text-muted">Result</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Review Status and Actions -->
<div class="row mb-4">
    <div class="col-md-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="fas fa-tasks me-2"></i>Review Status & Actions</h5>
            </div>
            <div class="card-body">
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <p class="mb-2">
                            <strong>Review Status:</strong>
                            <?php if ($result['reviewed_by']): ?>
                                <span class="badge bg-success">Reviewed</span>
                                <small class="text-muted">by <?php echo htmlspecialchars($result['reviewer_name']); ?> on <?php echo date('M d, Y H:i', strtotime($result['reviewed_at'])); ?></small>
                            <?php else: ?>
                                <span class="badge bg-warning">Pending Review</span>
                            <?php endif; ?>
                        </p>
                        <p class="mb-0">
                            <strong>Email Status:</strong>
                            <?php if ($result['email_sent']): ?>
                                <span class="badge bg-success">Sent</span>
                            <?php else: ?>
                                <span class="badge bg-secondary">Not Sent</span>
                            <?php endif; ?>
                        </p>
                    </div>
                    <div class="col-md-6 text-end">
                        <?php if (!$result['reviewed_by']): ?>
                            <form method="POST" class="d-inline">
                                <input type="hidden" name="action" value="mark_reviewed">
                                <button type="submit" class="btn btn-success btn-sm me-2">
                                    <i class="fas fa-check me-1"></i>Mark as Reviewed
                                </button>
                            </form>
                        <?php endif; ?>
                        
                        <?php if (!$result['email_sent']): ?>
                            <form method="POST" class="d-inline">
                                <input type="hidden" name="action" value="send_email">
                                <button type="submit" class="btn btn-primary btn-sm">
                                    <i class="fas fa-envelope me-1"></i>Mark Email Sent
                                </button>
                            </form>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Detailed Answers -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0"><i class="fas fa-list-alt me-2"></i>Detailed Answers</h5>
        <small class="text-muted">
            <?php if ($total_pages > 1): ?>
                Page <?php echo $page; ?> of <?php echo $total_pages; ?> (<?php echo $total_questions; ?> total questions)
            <?php else: ?>
                <?php echo $total_questions; ?> question<?php echo $total_questions != 1 ? 's' : ''; ?>
            <?php endif; ?>
        </small>
    </div>
    <div class="card-body">
        <?php foreach ($questions as $index => $question): ?>
            <div class="question-item mb-4 p-3 border rounded">
                <div class="d-flex justify-content-between align-items-start mb-3">
                    <h6 class="mb-0">Question <?php echo $offset + $index + 1; ?></h6>
                    <div>
                        <?php if ($question['selected_answer']): ?>
                            <?php if ($question['is_correct']): ?>
                                <span class="badge bg-success"><i class="fas fa-check"></i> Correct (+<?php echo $question['marks']; ?> marks)</span>
                            <?php else: ?>
                                <span class="badge bg-danger"><i class="fas fa-times"></i> Incorrect (0 marks)</span>
                            <?php endif; ?>
                        <?php else: ?>
                            <span class="badge bg-secondary"><i class="fas fa-minus"></i> Not Answered (0 marks)</span>
                        <?php endif; ?>
                    </div>
                </div>
                
                <p class="fw-bold mb-3"><?php echo htmlspecialchars($question['question']); ?></p>
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="options">
                            <?php 
                            $options = ['A' => $question['option_a'], 'B' => $question['option_b'], 'C' => $question['option_c'], 'D' => $question['option_d']];
                            foreach ($options as $key => $option): 
                                $class = '';
                                $icon = '';
                                
                                if ($key === $question['correct_answer']) {
                                    $class = 'text-success fw-bold';
                                    $icon = '<i class="fas fa-check-circle text-success me-2"></i>';
                                }
                                
                                if ($key === $question['selected_answer'] && $key !== $question['correct_answer']) {
                                    $class = 'text-danger fw-bold';
                                    $icon = '<i class="fas fa-times-circle text-danger me-2"></i>';
                                }
                                
                                if ($key === $question['selected_answer'] && $key === $question['correct_answer']) {
                                    $icon = '<i class="fas fa-check-circle text-success me-2"></i>';
                                }
                            ?>
                                <div class="option mb-2 <?php echo $class; ?>">
                                    <?php echo $icon; ?>
                                    <strong><?php echo $key; ?>.</strong> <?php echo htmlspecialchars($option); ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="answer-summary">
                            <p class="mb-1"><strong>Candidate's Answer:</strong> 
                                <?php if ($question['selected_answer']): ?>
                                    <span class="badge bg-info"><?php echo $question['selected_answer']; ?></span>
                                <?php else: ?>
                                    <span class="text-muted">Not answered</span>
                                <?php endif; ?>
                            </p>
                            <p class="mb-1"><strong>Correct Answer:</strong> 
                                <span class="badge bg-success"><?php echo $question['correct_answer']; ?></span>
                            </p>
                            <p class="mb-0"><strong>Marks:</strong> <?php echo $question['marks']; ?></p>
                        </div>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
        
        <!-- Pagination -->
        <?php if ($total_pages > 1): ?>
            <div class="d-flex justify-content-between align-items-center mt-4">
                <div>
                    <small class="text-muted">
                        Showing questions <?php echo $offset + 1; ?> to <?php echo min($offset + $questions_per_page, $total_questions); ?> of <?php echo $total_questions; ?> total questions
                    </small>
                </div>
                <nav aria-label="Questions pagination">
                    <ul class="pagination pagination-sm mb-0">
                        <?php if ($page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="?id=<?php echo $result_id; ?>&page=<?php echo $page - 1; ?>">
                                    <i class="fas fa-chevron-left"></i> Previous
                                </a>
                            </li>
                        <?php endif; ?>
                        
                        <?php
                        $start_page = max(1, $page - 2);
                        $end_page = min($total_pages, $page + 2);
                        
                        if ($start_page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="?id=<?php echo $result_id; ?>&page=1">1</a>
                            </li>
                            <?php if ($start_page > 2): ?>
                                <li class="page-item disabled">
                                    <span class="page-link">...</span>
                                </li>
                            <?php endif; ?>
                        <?php endif; ?>
                        
                        <?php for ($i = $start_page; $i <= $end_page; $i++): ?>
                            <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                <a class="page-link" href="?id=<?php echo $result_id; ?>&page=<?php echo $i; ?>"><?php echo $i; ?></a>
                            </li>
                        <?php endfor; ?>
                        
                        <?php if ($end_page < $total_pages): ?>
                            <?php if ($end_page < $total_pages - 1): ?>
                                <li class="page-item disabled">
                                    <span class="page-link">...</span>
                                </li>
                            <?php endif; ?>
                            <li class="page-item">
                                <a class="page-link" href="?id=<?php echo $result_id; ?>&page=<?php echo $total_pages; ?>"><?php echo $total_pages; ?></a>
                            </li>
                        <?php endif; ?>
                        
                        <?php if ($page < $total_pages): ?>
                            <li class="page-item">
                                <a class="page-link" href="?id=<?php echo $result_id; ?>&page=<?php echo $page + 1; ?>">
                                    Next <i class="fas fa-chevron-right"></i>
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </nav>
            </div>
        <?php endif; ?>
    </div>
</div>

<style>
.question-item {
    background-color: #f8f9fa;
}

.option {
    padding: 8px 12px;
    border-radius: 4px;
    background-color: white;
}

.answer-summary {
    background-color: white;
    padding: 15px;
    border-radius: 8px;
    border-left: 4px solid #007bff;
}
</style>

<?php include '../includes/footer.php'; ?>