<?php 
include '../includes/header.php'; 

if (!isAdmin()) {
    redirect('../login.php');
}

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $user_id = intval($_POST['user_id']);
    
    try {
        switch ($_POST['action']) {
            case 'approve':
                $stmt = $conn->prepare("UPDATE users SET status = 'approved', updated_at = NOW() WHERE id = ?");
                if ($stmt->execute([$user_id])) {
                    $_SESSION['success_message'] = "User approved successfully";
                } else {
                    $_SESSION['error_message'] = "Failed to approve user";
                }
                break;
                
            case 'reject':
                $stmt = $conn->prepare("UPDATE users SET status = 'rejected', updated_at = NOW() WHERE id = ?");
                if ($stmt->execute([$user_id])) {
                    $_SESSION['success_message'] = "User rejected successfully";
                } else {
                    $_SESSION['error_message'] = "Failed to reject user";
                }
                break;
                
            case 'delete':
                // Check if user has any test sessions
                $stmt = $conn->prepare("SELECT COUNT(*) FROM test_sessions WHERE user_id = ?");
                $stmt->execute([$user_id]);
                $has_sessions = $stmt->fetchColumn() > 0;
                
                if ($has_sessions) {
                    $_SESSION['error_message'] = "Cannot delete user with test sessions. Please contact administrator.";
                } else {
                    $stmt = $conn->prepare("DELETE FROM users WHERE id = ? AND role = 'candidate'");
                    if ($stmt->execute([$user_id])) {
                        $_SESSION['success_message'] = "User deleted successfully";
                    } else {
                        $_SESSION['error_message'] = "Failed to delete user";
                    }
                }
                break;
        }
    } catch (Exception $e) {
        $_SESSION['error_message'] = "Error: " . $e->getMessage();
    }
    
    header('Location: users.php');
    exit;
}

// Handle search and filter
$search = $_GET['search'] ?? '';
$status = $_GET['status'] ?? 'all';
$role = $_GET['role'] ?? 'candidate';

// Build query
$query = "SELECT * FROM users WHERE 1=1";
$params = [];

if ($role !== 'all') {
    $query .= " AND role = ?";
    $params[] = $role;
}

if ($status !== 'all') {
    $query .= " AND status = ?";
    $params[] = $status;
}

if ($search) {
    $query .= " AND (full_name LIKE ? OR email LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

$query .= " ORDER BY created_at DESC";

$stmt = $conn->prepare($query);
$stmt->execute($params);
$users = $stmt->fetchAll();

// Get statistics
$stats = $conn->query("SELECT 
    COUNT(*) as total,
    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
    SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
    SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected
    FROM users WHERE role = 'candidate'")->fetch();
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>
        <i class="fas fa-users me-2"></i>
        User Management
    </h2>
    <div class="d-flex gap-2">
        <button class="btn btn-outline-primary" onclick="exportUsers()">
            <i class="fas fa-download me-1"></i>Export
        </button>
    </div>
</div>

<!-- Statistics Cards -->
<div class="row mb-4">
    <div class="col-md-3">
        <div class="card">
            <div class="card-body text-center">
                <h4 class="text-primary"><?php echo $stats['total']; ?></h4>
                <p class="text-muted mb-0">Total Candidates</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card">
            <div class="card-body text-center">
                <h4 class="text-warning"><?php echo $stats['pending']; ?></h4>
                <p class="text-muted mb-0">Pending</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card">
            <div class="card-body text-center">
                <h4 class="text-success"><?php echo $stats['approved']; ?></h4>
                <p class="text-muted mb-0">Approved</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card">
            <div class="card-body text-center">
                <h4 class="text-danger"><?php echo $stats['rejected']; ?></h4>
                <p class="text-muted mb-0">Rejected</p>
            </div>
        </div>
    </div>
</div>

<!-- Search and Filter -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0"><i class="fas fa-filter me-2"></i>Search & Filter</h5>
    </div>
    <div class="card-body">
        <form method="GET">
            <div class="row">
                <div class="col-md-4 mb-3">
                    <label class="form-label">Search Users</label>
                    <input type="text" name="search" class="form-control" placeholder="Search by name or email..." value="<?php echo htmlspecialchars($search); ?>">
                </div>
                <div class="col-md-3 mb-3">
                    <label class="form-label">Status</label>
                    <select name="status" class="form-select">
                        <option value="all" <?php echo $status === 'all' ? 'selected' : ''; ?>>All Status</option>
                        <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>Pending</option>
                        <option value="approved" <?php echo $status === 'approved' ? 'selected' : ''; ?>>Approved</option>
                        <option value="rejected" <?php echo $status === 'rejected' ? 'selected' : ''; ?>>Rejected</option>
                    </select>
                </div>
                <div class="col-md-3 mb-3">
                    <label class="form-label">Role</label>
                    <select name="role" class="form-select">
                        <option value="candidate" <?php echo $role === 'candidate' ? 'selected' : ''; ?>>Candidates</option>
                        <option value="admin" <?php echo $role === 'admin' ? 'selected' : ''; ?>>Admins</option>
                        <option value="all" <?php echo $role === 'all' ? 'selected' : ''; ?>>All Users</option>
                    </select>
                </div>
                <div class="col-md-2 mb-3">
                    <label class="form-label">&nbsp;</label>
                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-search me-1"></i>Filter
                        </button>
                        <a href="users.php" class="btn btn-secondary">
                            <i class="fas fa-undo me-1"></i>Reset
                        </a>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Users Table -->
<div class="card">
    <div class="card-header">
        <h5 class="mb-0">
            <i class="fas fa-list me-2"></i>Users List (<?php echo count($users); ?> users)
        </h5>
    </div>
    <div class="card-body">
        <?php if (count($users) > 0): ?>
            <div class="table-responsive">
                <table class="table table-striped" id="usersTable">
                    <thead>
                        <tr>
                            <th>Full Name</th>
                            <th>Email</th>
                            <th>Role</th>
                            <th>Status</th>
                            <th>Registration Date</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($user['full_name']); ?></td>
                                <td><?php echo htmlspecialchars($user['email']); ?></td>
                                <td>
                                    <span class="badge bg-<?php echo $user['role'] === 'admin' ? 'danger' : 'primary'; ?>">
                                        <?php echo ucfirst($user['role']); ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="badge bg-<?php 
                                        echo $user['status'] === 'approved' ? 'success' : 
                                            ($user['status'] === 'rejected' ? 'danger' : 'warning'); ?>">
                                        <?php echo ucfirst($user['status']); ?>
                                    </span>
                                </td>
                                <td><?php echo date('M j, Y g:i A', strtotime($user['created_at'])); ?></td>
                                <td>
                                    <div class="btn-group" role="group">
                                        <?php if ($user['role'] === 'candidate' && $user['status'] === 'pending'): ?>
                                            <button class="btn btn-sm btn-success" onclick="approveUser(<?php echo $user['id']; ?>)" title="Approve User">
                                                <i class="fas fa-check"></i>
                                            </button>
                                            <button class="btn btn-sm btn-warning" onclick="rejectUser(<?php echo $user['id']; ?>)" title="Reject User">
                                                <i class="fas fa-times"></i>
                                            </button>
                                        <?php endif; ?>
                                        <button class="btn btn-sm btn-info" onclick="viewUser(<?php echo $user['id']; ?>)" title="View Details">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <?php if ($user['role'] === 'candidate'): ?>
                                            <button class="btn btn-sm btn-danger" onclick="deleteUser(<?php echo $user['id']; ?>)" title="Delete User">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <div class="text-center py-5">
                <i class="fas fa-users fa-3x text-muted mb-3"></i>
                <h5>No Users Found</h5>
                <p class="text-muted">No users match your current search criteria.</p>
                <a href="users.php" class="btn btn-primary">
                    <i class="fas fa-undo me-2"></i>Reset Filters
                </a>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Bottom spacing to prevent taskbar overlap -->
<style>
body {
    padding-bottom: 100px;
}

.card {
    margin-bottom: 50px;
}

@media (max-width: 768px) {
    body {
        padding-bottom: 120px;
    }
}
</style>

<script>
function approveUser(userId) {
    if (confirm('Are you sure you want to approve this user?')) {
        submitAction('approve', userId);
    }
}

function rejectUser(userId) {
    if (confirm('Are you sure you want to reject this user?')) {
        submitAction('reject', userId);
    }
}

function deleteUser(userId) {
    if (confirm('This action cannot be undone. Are you sure you want to delete this user?')) {
        submitAction('delete', userId);
    }
}

function submitAction(action, userId) {
    const form = document.createElement('form');
    form.method = 'POST';
    form.innerHTML = `
        <input type="hidden" name="action" value="${action}">
        <input type="hidden" name="user_id" value="${userId}">
    `;
    document.body.appendChild(form);
    form.submit();
}

function viewUser(userId) {
    Swal.fire({
        icon: 'info',
        title: 'User Information',
        text: `User ID: ${userId}\nDetailed user information would be displayed here.`,
        confirmButtonText: 'OK'
    });
}

function exportUsers() {
    const search = new URLSearchParams(window.location.search).get('search') || '';
    const status = new URLSearchParams(window.location.search).get('status') || 'all';
    const role = new URLSearchParams(window.location.search).get('role') || 'candidate';
    
    window.location.href = `export-users.php?search=${search}&status=${status}&role=${role}`;
}
</script>

<?php include '../includes/footer.php'; ?>