<?php 
include '../includes/header.php';
require_once '../includes/mail.php';
require_once '../includes/test-invitation-template.php';

if (!isAdmin()) {
    redirect('../login.php');
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create':
                $user_id = intval($_POST['user_id']);
                $job_role_id = intval($_POST['job_role_id']);
                $scheduled_time = $_POST['scheduled_time'];
                $duration_minutes = intval($_POST['duration_minutes']);
                $access_code = generateAccessCode();
                
                if (empty($user_id) || empty($job_role_id) || empty($scheduled_time)) {
                    $_SESSION['error_message'] = "All fields are required";
                } else {
                    $stmt = $conn->prepare("INSERT INTO test_sessions (user_id, job_role_id, access_code, scheduled_time, duration_minutes, created_by) VALUES (?, ?, ?, ?, ?, ?)");
                    if ($stmt->execute([$user_id, $job_role_id, $access_code, $scheduled_time, $duration_minutes, $_SESSION['user_id']])) {
                        
                        // Get candidate details
                        $candidate_stmt = $conn->prepare("SELECT full_name, email FROM users WHERE id = ?");
                        $candidate_stmt->execute([$user_id]);
                        $candidate = $candidate_stmt->fetch();
                        
                        // Get job role details
                        $role_stmt = $conn->prepare("SELECT role_name, description FROM job_roles WHERE id = ?");
                        $role_stmt->execute([$job_role_id]);
                        $job_role = $role_stmt->fetch();
                        
                        // Get question statistics for this job role
                        $stats_stmt = $conn->prepare("SELECT COUNT(*) as total_questions, SUM(marks) as total_marks FROM questions WHERE job_role_id = ?");
                        $stats_stmt->execute([$job_role_id]);
                        $stats = $stats_stmt->fetch();
                        
                        // Prepare session data for email
                        $sessionData = [
                            'access_code' => $access_code,
                            'scheduled_time' => $scheduled_time,
                            'duration_minutes' => $duration_minutes,
                            'total_questions' => $stats['total_questions'] ?? 0,
                            'total_marks' => $stats['total_marks'] ?? 0
                        ];
                        
                        // Send invitation email
                        try {
                            $mailHelper = new MailHelper();
                            $subject = "Test Invitation - " . $job_role['role_name'] . " | OACSOFT CBT";
                            
                            // Generate HTML email content
                            $htmlContent = generateTestInvitationEmail($candidate, $sessionData, $job_role);
                            
                            // Generate plain text content
                            $textContent = generateTestInvitationTextEmail($candidate, $sessionData, $job_role);
                            
                            $emailSent = $mailHelper->sendEmail(
                                $candidate['email'],
                                $candidate['full_name'],
                                $subject,
                                $htmlContent,
                                $textContent
                            );
                            
                            if ($emailSent) {
                                $_SESSION['success_message'] = "Test session created successfully! Invitation email sent to " . $candidate['email'] . ". Access Code: $access_code";
                            } else {
                                $_SESSION['success_message'] = "Test session created successfully. Access Code: $access_code (Note: Email could not be sent - please inform candidate manually)";
                            }
                            
                        } catch (Exception $e) {
                            error_log("Failed to send test invitation email: " . $e->getMessage());
                            $_SESSION['success_message'] = "Test session created successfully. Access Code: $access_code (Note: Email could not be sent - please inform candidate manually)";
                        }
                        
                    } else {
                        $_SESSION['error_message'] = "Failed to create test session";
                    }
                }
                break;
                
            case 'cancel':
                $id = $_POST['id'];
                $stmt = $conn->prepare("UPDATE test_sessions SET status = 'expired' WHERE id = ?");
                if ($stmt->execute([$id])) {
                    $_SESSION['success_message'] = "Test session cancelled successfully";
                } else {
                    $_SESSION['error_message'] = "Failed to cancel test session";
                }
                break;
        }
        header('Location: test-sessions.php');
        exit;
    }
}

// Fetch approved candidates
$stmt = $conn->query("SELECT id, full_name, email FROM users WHERE role = 'candidate' AND status = 'approved' ORDER BY full_name");
$candidates = $stmt->fetchAll();

// Fetch job roles
$stmt = $conn->query("SELECT id, role_name FROM job_roles ORDER BY role_name");
$job_roles = $stmt->fetchAll();

// Fetch test sessions with details
$stmt = $conn->query("SELECT ts.*, u.full_name, u.email, jr.role_name, creator.full_name as creator_name
                     FROM test_sessions ts
                     JOIN users u ON ts.user_id = u.id
                     JOIN job_roles jr ON ts.job_role_id = jr.id
                     JOIN users creator ON ts.created_by = creator.id
                     ORDER BY ts.created_at DESC");
$sessions = $stmt->fetchAll();
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>
        <i class="fas fa-play-circle me-2"></i>
        Test Sessions Management
    </h2>
    <div>
        <a href="preview-invitation-email.php" class="btn btn-outline-info me-2">
            <i class="fas fa-eye me-1"></i>Preview Email
        </a>
        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createSessionModal">
            <i class="fas fa-plus me-2"></i>Create New Session
        </button>
    </div>
</div>

<?php if (isset($_SESSION['success_message'])): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?php echo $_SESSION['success_message']; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php unset($_SESSION['success_message']); ?>
<?php endif; ?>

<?php if (isset($_SESSION['error_message'])): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?php echo $_SESSION['error_message']; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php unset($_SESSION['error_message']); ?>
<?php endif; ?>

<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Active Test Sessions</h5>
            </div>
            <div class="card-body">
                <?php if (count($sessions) > 0): ?>
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Candidate</th>
                                    <th>Job Role</th>
                                    <th>Access Code</th>
                                    <th>Scheduled Time</th>
                                    <th>Duration</th>
                                    <th>Status</th>
                                    <th>Created By</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($sessions as $session): ?>
                                    <tr>
                                        <td>
                                            <strong><?php echo htmlspecialchars($session['full_name']); ?></strong><br>
                                            <small class="text-muted"><?php echo htmlspecialchars($session['email']); ?></small>
                                        </td>
                                        <td><?php echo htmlspecialchars($session['role_name']); ?></td>
                                        <td>
                                            <span class="badge bg-primary fs-6">
                                                <?php echo $session['access_code']; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php echo date('M j, Y H:i', strtotime($session['scheduled_time'])); ?>
                                        </td>
                                        <td>
                                            <?php echo $session['duration_minutes']; ?> min
                                        </td>
                                        <td>
                                            <span class="badge status-<?php echo $session['status']; ?>">
                                                <?php echo ucfirst($session['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <small><?php echo htmlspecialchars($session['creator_name']); ?></small>
                                        </td>
                                        <td>
                                            <div class="action-buttons">
                                                <?php if ($session['status'] === 'scheduled'): ?>
                                                    <button class="btn btn-sm btn-danger" 
                                                            onclick="cancelSession(<?php echo $session['id']; ?>)">
                                                        <i class="fas fa-times"></i>
                                                    </button>
                                                <?php else: ?>
                                                    <span class="text-muted">-</span>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="text-center py-5">
                        <i class="fas fa-play-circle fa-3x text-muted mb-3"></i>
                        <h5 class="text-muted">No test sessions found</h5>
                        <p class="text-muted">Create your first test session to get started.</p>
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createSessionModal">
                            <i class="fas fa-plus me-1"></i>Create First Session
                        </button>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Create Session Modal -->
<div class="modal fade" id="createSessionModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Create Test Session</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="create">
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="user_id" class="form-label">Candidate *</label>
                        <select class="form-select" id="user_id" name="user_id" required>
                            <option value="">Select candidate</option>
                            <?php foreach ($candidates as $candidate): ?>
                                <option value="<?php echo $candidate['id']; ?>">
                                    <?php echo htmlspecialchars($candidate['full_name']); ?> 
                                    (<?php echo htmlspecialchars($candidate['email']); ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="job_role_id" class="form-label">Job Role *</label>
                        <select class="form-select" id="job_role_id" name="job_role_id" required>
                            <option value="">Select job role</option>
                            <?php foreach ($job_roles as $role): ?>
                                <option value="<?php echo $role['id']; ?>">
                                    <?php echo htmlspecialchars($role['role_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="scheduled_time" class="form-label">Scheduled Time *</label>
                        <input type="datetime-local" 
                               class="form-control" 
                               id="scheduled_time" 
                               name="scheduled_time" 
                               required
                               min="<?php echo date('Y-m-d\TH:i'); ?>">
                    </div>
                    
                    <div class="mb-3">
                        <label for="duration_minutes" class="form-label">Duration (minutes) *</label>
                        <input type="number" 
                               class="form-control" 
                               id="duration_minutes" 
                               name="duration_minutes" 
                               min="15" 
                               max="180" 
                               value="30" 
                               required>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Session</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Set default scheduled time to 1 hour from now
document.addEventListener('DOMContentLoaded', function() {
    const now = new Date();
    now.setHours(now.getHours() + 1);
    now.setMinutes(Math.ceil(now.getMinutes() / 15) * 15); // Round to next 15 minutes
    
    const year = now.getFullYear();
    const month = String(now.getMonth() + 1).padStart(2, '0');
    const day = String(now.getDate()).padStart(2, '0');
    const hours = String(now.getHours()).padStart(2, '0');
    const minutes = String(now.getMinutes()).padStart(2, '0');
    
    document.getElementById('scheduled_time').value = `${year}-${month}-${day}T${hours}:${minutes}`;
});

function cancelSession(id) {
    if (confirm('Are you sure you want to cancel this test session?')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
            <input type="hidden" name="action" value="cancel">
            <input type="hidden" name="id" value="${id}">
        `;
        document.body.appendChild(form);
        form.submit();
    }
}
</script>

<?php include '../includes/footer.php'; ?>