<?php 
include '../includes/header.php'; 

if (!isAdmin()) {
    redirect('../login.php');
}

// Handle actions
if ($_POST) {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'mark_reviewed':
                $result_id = $_POST['result_id'];
                $stmt = $conn->prepare("UPDATE test_results SET reviewed_by = ?, reviewed_at = NOW() WHERE id = ?");
                $stmt->execute([$_SESSION['user_id'], $result_id]);
                $_SESSION['success_message'] = 'Result marked as reviewed successfully!';
                break;
                
            case 'send_email':
                $result_id = $_POST['result_id'];
                $stmt = $conn->prepare("UPDATE test_results SET email_sent = TRUE WHERE id = ?");
                $stmt->execute([$result_id]);
                $_SESSION['success_message'] = 'Email status updated successfully!';
                break;
                
            case 'delete_result':
                $result_id = $_POST['result_id'];
                $stmt = $conn->prepare("DELETE FROM test_results WHERE id = ?");
                $stmt->execute([$result_id]);
                $_SESSION['success_message'] = 'Test result deleted successfully!';
                break;
        }
        header('Location: results.php');
        exit;
    }
}

// Get filter parameters
$search = $_GET['search'] ?? '';
$status_filter = $_GET['status'] ?? '';
$job_role_filter = $_GET['job_role'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$page = max(1, $_GET['page'] ?? 1);
$per_page = 20;
$offset = ($page - 1) * $per_page;

// Build query
$where_conditions = [];
$params = [];

if ($search) {
    $where_conditions[] = "(u.full_name LIKE ? OR u.email LIKE ? OR ts.access_code LIKE ?)";
    $search_param = "%$search%";
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
}

if ($status_filter) {
    if ($status_filter === 'reviewed') {
        $where_conditions[] = "tr.reviewed_by IS NOT NULL";
    } elseif ($status_filter === 'pending') {
        $where_conditions[] = "tr.reviewed_by IS NULL";
    } elseif ($status_filter === 'email_sent') {
        $where_conditions[] = "tr.email_sent = TRUE";
    } elseif ($status_filter === 'email_pending') {
        $where_conditions[] = "tr.email_sent = FALSE";
    }
}

if ($job_role_filter) {
    $where_conditions[] = "jr.id = ?";
    $params[] = $job_role_filter;
}

if ($date_from) {
    $where_conditions[] = "DATE(tr.completed_at) >= ?";
    $params[] = $date_from;
}

if ($date_to) {
    $where_conditions[] = "DATE(tr.completed_at) <= ?";
    $params[] = $date_to;
}

$where_clause = $where_conditions ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

// Get total count
$count_query = "SELECT COUNT(*) as total 
                FROM test_results tr 
                JOIN test_sessions ts ON tr.session_id = ts.id 
                JOIN users u ON ts.user_id = u.id 
                JOIN job_roles jr ON ts.job_role_id = jr.id 
                $where_clause";
$count_stmt = $conn->prepare($count_query);
$count_stmt->execute($params);
$total_results = $count_stmt->fetch()['total'];
$total_pages = ceil($total_results / $per_page);

// Get results
$query = "SELECT tr.*, u.full_name, u.email, jr.role_name, ts.access_code, ts.scheduled_time,
                 reviewer.full_name as reviewer_name
          FROM test_results tr 
          JOIN test_sessions ts ON tr.session_id = ts.id 
          JOIN users u ON ts.user_id = u.id 
          JOIN job_roles jr ON ts.job_role_id = jr.id 
          LEFT JOIN users reviewer ON tr.reviewed_by = reviewer.id
          $where_clause 
          ORDER BY tr.completed_at DESC 
          LIMIT $per_page OFFSET $offset";
$stmt = $conn->prepare($query);
$stmt->execute($params);
$results = $stmt->fetchAll();

// Get job roles for filter
$job_roles_stmt = $conn->query("SELECT * FROM job_roles ORDER BY role_name");
$job_roles = $job_roles_stmt->fetchAll();

// Get statistics
$stats_query = "SELECT 
    COUNT(*) as total_results,
    COUNT(CASE WHEN tr.reviewed_by IS NOT NULL THEN 1 END) as reviewed_count,
    COUNT(CASE WHEN tr.email_sent = TRUE THEN 1 END) as emails_sent,
    AVG(tr.percentage) as avg_percentage,
    COUNT(CASE WHEN tr.percentage >= 70 THEN 1 END) as passed_count
    FROM test_results tr";
$stats_stmt = $conn->query($stats_query);
$stats = $stats_stmt->fetch();
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h2><i class="fas fa-chart-bar me-2"></i>Test Results Management</h2>
    <div>
        <button class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#exportModal">
            <i class="fas fa-download me-2"></i>Export Results
        </button>
    </div>
</div>

<!-- Statistics Cards -->
<div class="row mb-4">
    <div class="col-md-2">
        <div class="card text-center">
            <div class="card-body">
                <h3 class="text-primary"><?php echo $stats['total_results']; ?></h3>
                <p class="mb-0">Total Results</p>
            </div>
        </div>
    </div>
    <div class="col-md-2">
        <div class="card text-center">
            <div class="card-body">
                <h3 class="text-success"><?php echo $stats['reviewed_count']; ?></h3>
                <p class="mb-0">Reviewed</p>
            </div>
        </div>
    </div>
    <div class="col-md-2">
        <div class="card text-center">
            <div class="card-body">
                <h3 class="text-info"><?php echo $stats['emails_sent']; ?></h3>
                <p class="mb-0">Emails Sent</p>
            </div>
        </div>
    </div>
    <div class="col-md-2">
        <div class="card text-center">
            <div class="card-body">
                <h3 class="text-warning"><?php echo number_format($stats['avg_percentage'], 1); ?>%</h3>
                <p class="mb-0">Avg Score</p>
            </div>
        </div>
    </div>
    <div class="col-md-2">
        <div class="card text-center">
            <div class="card-body">
                <h3 class="text-success"><?php echo $stats['passed_count']; ?></h3>
                <p class="mb-0">Passed</p>
            </div>
        </div>
    </div>
    <div class="col-md-2">
        <div class="card text-center">
            <div class="card-body">
                <h3 class="text-danger"><?php echo $stats['total_results'] - $stats['passed_count']; ?></h3>
                <p class="mb-0">Failed</p>
            </div>
        </div>
    </div>
</div>

<!-- Filters -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0"><i class="fas fa-filter me-2"></i>Filters</h5>
    </div>
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-3">
                <label class="form-label">Search</label>
                <input type="text" class="form-control" name="search" value="<?php echo htmlspecialchars($search); ?>" placeholder="Name, email, or access code">
            </div>
            <div class="col-md-2">
                <label class="form-label">Status</label>
                <select class="form-select" name="status">
                    <option value="">All Status</option>
                    <option value="reviewed" <?php echo $status_filter === 'reviewed' ? 'selected' : ''; ?>>Reviewed</option>
                    <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending Review</option>
                    <option value="email_sent" <?php echo $status_filter === 'email_sent' ? 'selected' : ''; ?>>Email Sent</option>
                    <option value="email_pending" <?php echo $status_filter === 'email_pending' ? 'selected' : ''; ?>>Email Pending</option>
                </select>
            </div>
            <div class="col-md-2">
                <label class="form-label">Job Role</label>
                <select class="form-select" name="job_role">
                    <option value="">All Roles</option>
                    <?php foreach ($job_roles as $role): ?>
                        <option value="<?php echo $role['id']; ?>" <?php echo $job_role_filter == $role['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($role['role_name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-2">
                <label class="form-label">Date From</label>
                <input type="date" class="form-control" name="date_from" value="<?php echo $date_from; ?>">
            </div>
            <div class="col-md-2">
                <label class="form-label">Date To</label>
                <input type="date" class="form-control" name="date_to" value="<?php echo $date_to; ?>">
            </div>
            <div class="col-md-1">
                <label class="form-label">&nbsp;</label>
                <button type="submit" class="btn btn-primary d-block w-100">
                    <i class="fas fa-search"></i>
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Results Table -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">Test Results (<?php echo $total_results; ?> total)</h5>
        <small class="text-muted">Page <?php echo $page; ?> of <?php echo $total_pages; ?></small>
    </div>
    <div class="card-body p-0">
        <?php if (count($results) > 0): ?>
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead class="table-light">
                        <tr>
                            <th>Candidate</th>
                            <th>Job Role</th>
                            <th>Access Code</th>
                            <th>Score</th>
                            <th>Percentage</th>
                            <th>Status</th>
                            <th>Completed</th>
                            <th>Review Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($results as $result): 
                            $passed = $result['percentage'] >= 70;
                            $is_reviewed = !empty($result['reviewed_by']);
                            $email_sent = $result['email_sent'];
                        ?>
                            <tr>
                                <td>
                                    <div>
                                        <strong><?php echo htmlspecialchars($result['full_name']); ?></strong><br>
                                        <small class="text-muted"><?php echo htmlspecialchars($result['email']); ?></small>
                                    </div>
                                </td>
                                <td><?php echo htmlspecialchars($result['role_name']); ?></td>
                                <td><code><?php echo htmlspecialchars($result['access_code']); ?></code></td>
                                <td><?php echo $result['correct_answers']; ?>/<?php echo $result['total_questions']; ?></td>
                                <td>
                                    <span class="badge bg-<?php echo $passed ? 'success' : 'danger'; ?>">
                                        <?php echo number_format($result['percentage'], 1); ?>%
                                    </span>
                                </td>
                                <td>
                                    <span class="badge bg-<?php echo $passed ? 'success' : 'danger'; ?>">
                                        <i class="fas fa-<?php echo $passed ? 'check' : 'times'; ?> me-1"></i>
                                        <?php echo $passed ? 'PASSED' : 'FAILED'; ?>
                                    </span>
                                </td>
                                <td>
                                    <small><?php echo date('M j, Y g:i A', strtotime($result['completed_at'])); ?></small>
                                </td>
                                <td>
                                    <?php if ($is_reviewed): ?>
                                        <span class="badge bg-success">
                                            <i class="fas fa-check me-1"></i>Reviewed
                                        </span><br>
                                        <small class="text-muted">by <?php echo htmlspecialchars($result['reviewer_name']); ?></small>
                                    <?php else: ?>
                                        <span class="badge bg-warning">
                                            <i class="fas fa-clock me-1"></i>Pending
                                        </span>
                                    <?php endif; ?>
                                    <br>
                                    <?php if ($email_sent): ?>
                                        <span class="badge bg-info mt-1">
                                            <i class="fas fa-envelope me-1"></i>Email Sent
                                        </span>
                                    <?php else: ?>
                                        <span class="badge bg-secondary mt-1">
                                            <i class="fas fa-envelope-open me-1"></i>Email Pending
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div class="btn-group" role="group">
                                        <a href="view-result.php?id=<?php echo $result['id']; ?>" 
                                           class="btn btn-sm btn-outline-primary" title="View Details">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        
                                        <?php if (!$is_reviewed): ?>
                                            <form method="POST" class="d-inline">
                                                <input type="hidden" name="action" value="mark_reviewed">
                                                <input type="hidden" name="result_id" value="<?php echo $result['id']; ?>">
                                                <button type="submit" class="btn btn-sm btn-outline-success" title="Mark as Reviewed">
                                                    <i class="fas fa-check"></i>
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                        
                                        <?php if (!$email_sent): ?>
                                            <form method="POST" class="d-inline">
                                                <input type="hidden" name="action" value="send_email">
                                                <input type="hidden" name="result_id" value="<?php echo $result['id']; ?>">
                                                <button type="submit" class="btn btn-sm btn-outline-info" title="Mark Email as Sent">
                                                    <i class="fas fa-envelope"></i>
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                        
                                        <form method="POST" class="d-inline" onsubmit="return confirm('Are you sure you want to delete this result?')">
                                            <input type="hidden" name="action" value="delete_result">
                                            <input type="hidden" name="result_id" value="<?php echo $result['id']; ?>">
                                            <button type="submit" class="btn btn-sm btn-outline-danger" title="Delete Result">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
                <div class="card-footer">
                    <nav>
                        <ul class="pagination justify-content-center mb-0">
                            <?php if ($page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>">
                                        <i class="fas fa-chevron-left"></i>
                                    </a>
                                </li>
                            <?php endif; ?>
                            
                            <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                                <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                                    <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>">
                                        <?php echo $i; ?>
                                    </a>
                                </li>
                            <?php endfor; ?>
                            
                            <?php if ($page < $total_pages): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>">
                                        <i class="fas fa-chevron-right"></i>
                                    </a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                </div>
            <?php endif; ?>
        <?php else: ?>
            <div class="text-center py-5">
                <i class="fas fa-chart-bar fa-3x text-muted mb-3"></i>
                <h5 class="text-muted">No test results found</h5>
                <p class="text-muted">No test results match your current filters.</p>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Export Modal -->
<div class="modal fade" id="exportModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Export Test Results</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Choose export format and options:</p>
                <form>
                    <div class="mb-3">
                        <label class="form-label">Export Format</label>
                        <select class="form-select">
                            <option value="csv">CSV (Comma Separated Values)</option>
                            <option value="excel">Excel Spreadsheet</option>
                            <option value="pdf">PDF Report</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Include</label>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" checked>
                            <label class="form-check-label">Candidate Information</label>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" checked>
                            <label class="form-check-label">Test Scores</label>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox">
                            <label class="form-check-label">Detailed Answers</label>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary">Export Results</button>
            </div>
        </div>
    </div>
</div>

<?php include '../includes/footer.php'; ?>