<?php 
require_once '../config/database.php';

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

if (!isAdmin()) {
    redirect('../login.php');
}

// Get job role ID from URL
$job_role_id = isset($_GET['role']) ? intval($_GET['role']) : 0;

// Initialize database connection
$db = new Database();
$conn = $db->connect();

// Fetch job role details
$stmt = $conn->prepare("SELECT * FROM job_roles WHERE id = ?");
$stmt->execute([$job_role_id]);
$job_role = $stmt->fetch();

if (!$job_role) {
    $_SESSION['error_message'] = "Job role not found";
    header('Location: job-roles.php');
    exit;
}

include '../includes/header.php';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'add':
                $question = sanitize($_POST['question']);
                $option_a = sanitize($_POST['option_a']);
                $option_b = sanitize($_POST['option_b']);
                $option_c = sanitize($_POST['option_c']);
                $option_d = sanitize($_POST['option_d']);
                $correct_answer = $_POST['correct_answer'];
                $marks = intval($_POST['marks']);
                
                if (empty($question) || empty($option_a) || empty($option_b) || empty($option_c) || empty($option_d)) {
                    $_SESSION['error_message'] = "All fields are required";
                } else {
                    $stmt = $conn->prepare("INSERT INTO questions (job_role_id, question, option_a, option_b, option_c, option_d, correct_answer, marks) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                    if ($stmt->execute([$job_role_id, $question, $option_a, $option_b, $option_c, $option_d, $correct_answer, $marks])) {
                        $_SESSION['success_message'] = "Question added successfully";
                    } else {
                        $_SESSION['error_message'] = "Failed to add question";
                    }
                }
                break;
                
            case 'edit':
                $id = $_POST['id'];
                $question = sanitize($_POST['question']);
                $option_a = sanitize($_POST['option_a']);
                $option_b = sanitize($_POST['option_b']);
                $option_c = sanitize($_POST['option_c']);
                $option_d = sanitize($_POST['option_d']);
                $correct_answer = $_POST['correct_answer'];
                $marks = intval($_POST['marks']);
                
                if (empty($question) || empty($option_a) || empty($option_b) || empty($option_c) || empty($option_d)) {
                    $_SESSION['error_message'] = "All fields are required";
                } else {
                    $stmt = $conn->prepare("UPDATE questions SET question = ?, option_a = ?, option_b = ?, option_c = ?, option_d = ?, correct_answer = ?, marks = ? WHERE id = ?");
                    if ($stmt->execute([$question, $option_a, $option_b, $option_c, $option_d, $correct_answer, $marks, $id])) {
                        $_SESSION['success_message'] = "Question updated successfully";
                    } else {
                        $_SESSION['error_message'] = "Failed to update question";
                    }
                }
                break;
                
            case 'delete':
                $id = $_POST['id'];
                $stmt = $conn->prepare("DELETE FROM questions WHERE id = ?");
                if ($stmt->execute([$id])) {
                    $_SESSION['success_message'] = "Question deleted successfully";
                } else {
                    $_SESSION['error_message'] = "Failed to delete question";
                }
                break;
        }
        header("Location: questions.php?role=$job_role_id");
        exit;
    }
}

// Fetch all questions for this job role
$stmt = $conn->prepare("SELECT * FROM questions WHERE job_role_id = ? ORDER BY id");
$stmt->execute([$job_role_id]);
$questions = $stmt->fetchAll();

// Get question for editing
$edit_question = null;
if (isset($_GET['edit'])) {
    $stmt = $conn->prepare("SELECT * FROM questions WHERE id = ? AND job_role_id = ?");
    $stmt->execute([$_GET['edit'], $job_role_id]);
    $edit_question = $stmt->fetch();
}
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h2>
            <i class="fas fa-question-circle me-2"></i>
            Questions for <?php echo htmlspecialchars($job_role['role_name']); ?>
        </h2>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item"><a href="job-roles.php">Job Roles</a></li>
                <li class="breadcrumb-item active">Questions</li>
            </ol>
        </nav>
    </div>
    <div>
        <a href="job-roles.php" class="btn btn-secondary">
            <i class="fas fa-arrow-left me-1"></i>Back to Roles
        </a>
        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addQuestionModal">
            <i class="fas fa-plus me-1"></i>Add Question
        </button>
    </div>
</div>

<?php if (isset($_SESSION['success_message'])): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?php echo $_SESSION['success_message']; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php unset($_SESSION['success_message']); ?>
<?php endif; ?>

<?php if (isset($_SESSION['error_message'])): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?php echo $_SESSION['error_message']; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php unset($_SESSION['error_message']); ?>
<?php endif; ?>

<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">Questions (<?php echo count($questions); ?>)</h5>
        <div>
            <span class="badge bg-info"><?php echo count($questions); ?> questions</span>
        </div>
    </div>
    <div class="card-body">
        <?php if (count($questions) > 0): ?>
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th style="width: 5%">#</th>
                            <th style="width: 40%">Question</th>
                            <th style="width: 30%">Options</th>
                            <th style="width: 10%">Answer</th>
                            <th style="width: 5%">Marks</th>
                            <th style="width: 10%">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($questions as $index => $question): ?>
                            <tr>
                                <td><?php echo $index + 1; ?></td>
                                <td>
                                    <div class="text-wrap" style="max-width: 300px;">
                                        <?php echo htmlspecialchars(substr($question['question'], 0, 100)); ?>
                                        <?php echo strlen($question['question']) > 100 ? '...' : ''; ?>
                                    </div>
                                </td>
                                <td>
                                    <small>
                                        A: <?php echo htmlspecialchars(substr($question['option_a'], 0, 20)); ?>...<br>
                                        B: <?php echo htmlspecialchars(substr($question['option_b'], 0, 20)); ?>...<br>
                                        C: <?php echo htmlspecialchars(substr($question['option_c'], 0, 20)); ?>...<br>
                                        D: <?php echo htmlspecialchars(substr($question['option_d'], 0, 20)); ?>...
                                    </small>
                                </td>
                                <td>
                                    <span class="badge bg-success">
                                        <?php echo $question['correct_answer']; ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="badge bg-info">
                                        <?php echo $question['marks']; ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="action-buttons">
                                        <button class="btn btn-sm btn-warning" 
                                                onclick="editQuestion(<?php echo $question['id']; ?>)"
                                                title="Edit">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button class="btn btn-sm btn-danger" 
                                                onclick="deleteQuestion(<?php echo $question['id']; ?>)"
                                                title="Delete">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <div class="text-center py-5">
                <i class="fas fa-question-circle fa-3x text-muted mb-3"></i>
                <h5 class="text-muted">No questions found</h5>
                <p class="text-muted">Add your first question to get started.</p>
                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addQuestionModal">
                    <i class="fas fa-plus me-1"></i>Add First Question
                </button>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Add Question Modal -->
<div class="modal fade" id="addQuestionModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add New Question</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="add">
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="question" class="form-label">Question *</label>
                        <textarea class="form-control" 
                                  id="question" 
                                  name="question" 
                                  rows="3" 
                                  required></textarea>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="option_a" class="form-label">Option A *</label>
                            <input type="text" 
                                   class="form-control" 
                                   id="option_a" 
                                   name="option_a" 
                                   required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="option_b" class="form-label">Option B *</label>
                            <input type="text" 
                                   class="form-control" 
                                   id="option_b" 
                                   name="option_b" 
                                   required>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="option_c" class="form-label">Option C *</label>
                            <input type="text" 
                                   class="form-control" 
                                   id="option_c" 
                                   name="option_c" 
                                   required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="option_d" class="form-label">Option D *</label>
                            <input type="text" 
                                   class="form-control" 
                                   id="option_d" 
                                   name="option_d" 
                                   required>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="correct_answer" class="form-label">Correct Answer *</label>
                            <select class="form-select" id="correct_answer" name="correct_answer" required>
                                <option value="">Select correct answer</option>
                                <option value="A">Option A</option>
                                <option value="B">Option B</option>
                                <option value="C">Option C</option>
                                <option value="D">Option D</option>
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="marks" class="form-label">Marks *</label>
                            <input type="number" 
                                   class="form-control" 
                                   id="marks" 
                                   name="marks" 
                                   min="1" 
                                   value="1" 
                                   required>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Add Question</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Question Modal -->
<?php if ($edit_question): ?>
<div class="modal fade" id="editQuestionModal" tabindex="-1" style="display: none;">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Edit Question</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="edit">
                <input type="hidden" name="id" value="<?php echo $edit_question['id']; ?>">
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="edit_question" class="form-label">Question *</label>
                        <textarea class="form-control" 
                                  id="edit_question" 
                                  name="question" 
                                  rows="3" 
                                  required><?php echo htmlspecialchars($edit_question['question']); ?></textarea>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="edit_option_a" class="form-label">Option A *</label>
                            <input type="text" 
                                   class="form-control" 
                                   id="edit_option_a" 
                                   name="option_a" 
                                   value="<?php echo htmlspecialchars($edit_question['option_a']); ?>"
                                   required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="edit_option_b" class="form-label">Option B *</label>
                            <input type="text" 
                                   class="form-control" 
                                   id="edit_option_b" 
                                   name="option_b" 
                                   value="<?php echo htmlspecialchars($edit_question['option_b']); ?>"
                                   required>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="edit_option_c" class="form-label">Option C *</label>
                            <input type="text" 
                                   class="form-control" 
                                   id="edit_option_c" 
                                   name="option_c" 
                                   value="<?php echo htmlspecialchars($edit_question['option_c']); ?>"
                                   required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="edit_option_d" class="form-label">Option D *</label>
                            <input type="text" 
                                   class="form-control" 
                                   id="edit_option_d" 
                                   name="option_d" 
                                   value="<?php echo htmlspecialchars($edit_question['option_d']); ?>"
                                   required>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="edit_correct_answer" class="form-label">Correct Answer *</label>
                            <select class="form-select" id="edit_correct_answer" name="correct_answer" required>
                                <option value="A" <?php echo $edit_question['correct_answer'] === 'A' ? 'selected' : ''; ?>>Option A</option>
                                <option value="B" <?php echo $edit_question['correct_answer'] === 'B' ? 'selected' : ''; ?>>Option B</option>
                                <option value="C" <?php echo $edit_question['correct_answer'] === 'C' ? 'selected' : ''; ?>>Option C</option>
                                <option value="D" <?php echo $edit_question['correct_answer'] === 'D' ? 'selected' : ''; ?>>Option D</option>
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="edit_marks" class="form-label">Marks *</label>
                            <input type="number" 
                                   class="form-control" 
                                   id="edit_marks" 
                                   name="marks" 
                                   min="1" 
                                   value="<?php echo $edit_question['marks']; ?>"
                                   required>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Update Question</button>
                </div>
            </form>
        </div>
    </div>
</div>
<?php endif; ?>

<script>
<?php if ($edit_question): ?>
document.addEventListener('DOMContentLoaded', function() {
    new bootstrap.Modal(document.getElementById('editQuestionModal')).show();
});
<?php endif; ?>

function editQuestion(id) {
    window.location.href = `questions.php?role=<?php echo $job_role_id; ?>&edit=${id}`;
}

function deleteQuestion(id) {
    if (confirm('Are you sure you want to delete this question? This action cannot be undone.')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
            <input type="hidden" name="action" value="delete">
            <input type="hidden" name="id" value="${id}">
        `;
        document.body.appendChild(form);
        form.submit();
    }
}
</script>

<?php include '../includes/footer.php'; ?>