<?php
session_start();
require_once '../../config/database.php';
require_once '../../includes/mail.php';

// Check if user is admin
if (!isAdmin()) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Access denied']);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get the result ID from POST data
$input = json_decode(file_get_contents('php://input'), true);
$result_id = isset($input['result_id']) ? intval($input['result_id']) : 0;

if ($result_id <= 0) {
    echo json_encode(['success' => false, 'message' => 'Invalid result ID']);
    exit;
}

try {
    $db = new Database();
    $conn = $db->connect();
    
    // Check if result exists and email hasn't been sent yet
    $stmt = $conn->prepare("SELECT email_sent FROM test_results WHERE id = ?");
    $stmt->execute([$result_id]);
    $result = $stmt->fetch();
    
    if (!$result) {
        echo json_encode(['success' => false, 'message' => 'Test result not found']);
        exit;
    }
    
    if ($result['email_sent']) {
        echo json_encode(['success' => false, 'message' => 'Email has already been sent for this result']);
        exit;
    }
    
    // Get test result details for email
    $stmt = $conn->prepare("
        SELECT 
            tr.*,
            ts.title as test_title,
            u.full_name,
            u.email,
            jr.role_name
        FROM test_results tr
        JOIN test_sessions ts ON tr.session_id = ts.id
        JOIN users u ON ts.user_id = u.id
        JOIN job_roles jr ON ts.job_role_id = jr.id
        WHERE tr.id = ?
    ");
    $stmt->execute([$result_id]);
    $result_details = $stmt->fetch();
    
    if (!$result_details) {
        echo json_encode(['success' => false, 'message' => 'Test result details not found']);
        exit;
    }
    
    // Prepare result data for email
    $email_data = [
        'test_title' => $result_details['test_title'] ?? $result_details['role_name'],
        'score' => round(($result_details['score'] / $result_details['total_questions']) * 100, 1),
        'correct_answers' => $result_details['correct_answers'],
        'total_questions' => $result_details['total_questions'],
        'completed_at' => $result_details['completed_at']
    ];
    
    // Send email using MailHelper
    $mailHelper = new MailHelper();
    $email_result = $mailHelper->sendResultEmail(
        $result_details['email'], 
        $result_details['full_name'], 
        $email_data
    );
    
    if ($email_result['success']) {
        // Mark email as sent in database
        $stmt = $conn->prepare("UPDATE test_results SET email_sent = TRUE WHERE id = ?");
        $stmt->execute([$result_id]);
        
        echo json_encode([
            'success' => true, 
            'message' => 'Test result email sent successfully!'
        ]);
    } else {
        echo json_encode([
            'success' => false, 
            'message' => 'Failed to send email: ' . $email_result['message']
        ]);
    }
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false, 
        'message' => 'Server error: ' . $e->getMessage()
    ]);
}
?>